package com.github.xuejike.query.dataverse.auth;

import com.microsoft.aad.msal4j.ClientCredentialFactory;
import com.microsoft.aad.msal4j.ClientCredentialParameters;
import com.microsoft.aad.msal4j.ConfidentialClientApplication;
import com.microsoft.aad.msal4j.IAuthenticationResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.Date;
import java.util.concurrent.CompletableFuture;

/**
 * OAuth 2.0 authentication provider for Dataverse using MSAL4J.
 * Supports client credentials flow (service principal authentication).
 *
 * @author xuejike
 */
public class OAuth2AuthProvider implements DataverseAuthProvider {
    
    private static final Logger log = LoggerFactory.getLogger(OAuth2AuthProvider.class);
    
    private final String clientId;
    private final String clientSecret;
    private final String tenantId;
    private final String scope;
    private final TokenCache tokenCache;
    
    private ConfidentialClientApplication app;
    
    /**
     * Create a new OAuth2AuthProvider.
     *
     * @param clientId the Azure AD application (client) ID
     * @param clientSecret the client secret
     * @param tenantId the Azure AD tenant ID
     * @param scope the scope for the access token (e.g., "https://org.crm.dynamics.com/.default")
     * @param tokenCache the token cache for storing tokens
     */
    public OAuth2AuthProvider(String clientId, String clientSecret, String tenantId, 
                              String scope, TokenCache tokenCache) {
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.tenantId = tenantId;
        this.scope = scope;
        this.tokenCache = tokenCache;
        
        initializeApp();
    }
    
    private void initializeApp() {
        try {
            String authority = "https://login.microsoftonline.com/" + tenantId;
            this.app = ConfidentialClientApplication.builder(
                    clientId,
                    ClientCredentialFactory.createFromSecret(clientSecret))
                    .authority(authority)
                    .build();
        } catch (Exception e) {
            log.error("Failed to initialize MSAL application", e);
            throw new RuntimeException("Failed to initialize OAuth2 authentication", e);
        }
    }
    
    @Override
    public String getAccessToken() {
        // Check cache first
        if (tokenCache != null && tokenCache.hasValidToken()) {
            return tokenCache.getAccessToken();
        }
        
        // Acquire new token
        return acquireToken();
    }
    
    @Override
    public String refreshToken() {
        return acquireToken();
    }
    
    @Override
    public boolean isTokenExpired() {
        if (tokenCache == null) {
            return true;
        }
        return !tokenCache.hasValidToken();
    }
    
    private String acquireToken() {
        try {
            ClientCredentialParameters parameters = ClientCredentialParameters.builder(
                    Collections.singleton(scope))
                    .build();
            
            CompletableFuture<IAuthenticationResult> future = app.acquireToken(parameters);
            IAuthenticationResult result = future.get();
            
            String accessToken = result.accessToken();
            Date expiresOn = result.expiresOnDate();
            
            // Cache the token
            if (tokenCache != null) {
                tokenCache.setAccessToken(accessToken, expiresOn);
            }
            
            log.debug("Successfully acquired access token, expires on: {}", expiresOn);
            return accessToken;
            
        } catch (Exception e) {
            log.error("Failed to acquire access token", e);
            throw new RuntimeException("Failed to acquire access token", e);
        }
    }
}
