package com.github.xuejike.query.dataverse.auth;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;

/**
 * Simple in-memory cache for storing OAuth access tokens.
 * Handles token expiration checking with a buffer period.
 *
 * @author xuejike
 */
public class TokenCache {
    
    private static final Logger log = LoggerFactory.getLogger(TokenCache.class);
    
    private String accessToken;
    private Date expiresOn;
    
    // Buffer time in milliseconds before actual expiration (default 5 minutes)
    private long expirationBufferMs = 5 * 60 * 1000;
    
    /**
     * Create a new TokenCache with default expiration buffer.
     */
    public TokenCache() {
    }
    
    /**
     * Create a new TokenCache with custom expiration buffer.
     *
     * @param expirationBufferMs buffer time in milliseconds before token expiration
     */
    public TokenCache(long expirationBufferMs) {
        this.expirationBufferMs = expirationBufferMs;
    }
    
    /**
     * Set the access token and its expiration time.
     *
     * @param accessToken the access token
     * @param expiresOn the expiration time
     */
    public void setAccessToken(String accessToken, Date expiresOn) {
        this.accessToken = accessToken;
        this.expiresOn = expiresOn;
        log.debug("Token cached, expires on: {}", expiresOn);
    }
    
    /**
     * Check if the cache has a valid (non-expired) token.
     *
     * @return true if a valid token exists
     */
    public boolean hasValidToken() {
        if (accessToken == null || expiresOn == null) {
            return false;
        }
        
        // Check if token is expired (with buffer)
        long now = System.currentTimeMillis();
        long expirationTime = expiresOn.getTime() - expirationBufferMs;
        
        boolean isValid = now < expirationTime;
        if (!isValid) {
            log.debug("Token expired or about to expire");
        }
        
        return isValid;
    }
    
    /**
     * Clear the cached token.
     */
    public void clear() {
        this.accessToken = null;
        this.expiresOn = null;
        log.debug("Token cache cleared");
    }
    
    /**
     * Get the access token.
     *
     * @return the access token
     */
    public String getAccessToken() {
        return accessToken;
    }
    
    /**
     * Get the expiration time.
     *
     * @return the expiration time
     */
    public Date getExpiresOn() {
        return expiresOn;
    }
    
    /**
     * Get the expiration buffer in milliseconds.
     *
     * @return the expiration buffer
     */
    public long getExpirationBufferMs() {
        return expirationBufferMs;
    }
    
    /**
     * Set the expiration buffer in milliseconds.
     *
     * @param expirationBufferMs the expiration buffer
     */
    public void setExpirationBufferMs(long expirationBufferMs) {
        this.expirationBufferMs = expirationBufferMs;
    }
}
