package com.github.xuejike.query.dataverse.client;

import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpResponse;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.github.xuejike.query.core.exception.LambdaQueryException;
import com.github.xuejike.query.dataverse.auth.DataverseAuthProvider;
import com.github.xuejike.query.dataverse.exception.DataverseAuthException;
import com.github.xuejike.query.dataverse.exception.DataverseConnectionException;
import com.github.xuejike.query.dataverse.exception.DataverseQueryException;
import com.github.xuejike.query.dataverse.exception.EntityNotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.UnsupportedEncodingException;
import java.net.SocketTimeoutException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;

/**
 * HTTP client for making requests to Microsoft Dataverse OData API.
 * Handles authentication, request building, and response parsing.
 *
 * @author xuejike
 */
public class DataverseClient {
    
    private static final Logger log = LoggerFactory.getLogger(DataverseClient.class);
    
    private final String baseUrl;
    private final DataverseAuthProvider authProvider;
    
    /**
     * Create a new DataverseClient.
     *
     * @param environmentUrl the Dataverse environment URL (e.g., "https://org.crm.dynamics.com")
     * @param authProvider the authentication provider
     */
    public DataverseClient(String environmentUrl, DataverseAuthProvider authProvider) {
        this.baseUrl = environmentUrl + "/api/data/v9.2/";
        this.authProvider = authProvider;
    }
    
    /**
     * Execute an OData query.
     *
     * @param query the OData query
     * @param nextLink optional next link for pagination (overrides query)
     * @return the OData response
     */
    public ODataResponse execute(ODataQuery query, String nextLink) {
        String url = nextLink != null ? nextLink : buildUrl(query);
        return executeGet(url);
    }
    
    /**
     * Get a single entity by its path.
     *
     * @param entityPath the entity path (e.g., "accounts(id)")
     * @return the OData response
     */
    public ODataResponse get(String entityPath) {
        String url = baseUrl + entityPath;
        return executeGet(url);
    }
    
    private ODataResponse executeGet(String url) {
        return executeGetWithRetry(url, false);
    }
    
    /**
     * 执行GET请求，支持令牌过期重试
     *
     * @param url 请求URL
     * @param isRetry 是否为重试请求
     * @return OData响应
     */
    private ODataResponse executeGetWithRetry(String url, boolean isRetry) {
        try {
            String accessToken = authProvider.getAccessToken();
            
            HttpRequest request = HttpRequest.get(url)
                    .header("Authorization", "Bearer " + accessToken)
                    .header("Accept", "application/json")
                    .header("OData-MaxVersion", "4.0")
                    .header("OData-Version", "4.0")
                    .header("Prefer", "odata.include-annotations=\"OData.Community.Display.V1.FormattedValue\"");
            
            log.debug("Executing Dataverse request: {}", url);
            log.debug("Request Token {}",accessToken);
            HttpResponse response = request.execute();
            
            if (!response.isOk()) {
                // 检查是否为401错误且未重试过
                if (response.getStatus() == 401 && !isRetry) {
                    log.info("Token expired, attempting to refresh and retry");
                    // 令牌过期，尝试刷新并重试
                    authProvider.refreshToken();
                    return executeGetWithRetry(url, true);
                }
                handleError(response, url);
            }
            
            return parseResponse(response.body());
            
        } catch (Exception e) {
            // 检查是否为超时异常
            if (e.getCause() instanceof SocketTimeoutException || 
                e.getMessage() != null && e.getMessage().contains("timeout")) {
                log.error("Connection timeout: {}", url, e);
                throw new DataverseConnectionException("Connection timeout: " + url, e);
            }
            
            if (e instanceof LambdaQueryException) {
                throw (LambdaQueryException) e;
            }
            log.error("Dataverse query failed: {}", url, e);
            throw new DataverseConnectionException("Dataverse query failed: " + url, e);
        }
    }
    
    /**
     * 构建OData查询URL
     *
     * @param query OData查询对象
     * @return 完整的查询URL
     */
    private String buildUrl(ODataQuery query) {
        StringBuilder url = new StringBuilder(baseUrl);
        url.append(query.getEntityName());
        
        List<String> params = new ArrayList<>();
        
        // 添加$filter参数
        if (StrUtil.isNotBlank(query.getFilter())) {
            try {
                params.add("$filter=" + URLEncoder.encode(query.getFilter(), "UTF-8"));
            } catch (UnsupportedEncodingException e) {
                throw new LambdaQueryException("Failed to encode filter parameter", e);
            }
        }
        
        // 添加$select参数
        if (StrUtil.isNotBlank(query.getSelect())) {
            params.add("$select=" + query.getSelect());
        }
        
        // 添加$orderby参数
        if (StrUtil.isNotBlank(query.getOrderBy())) {
            params.add("$orderby=" + query.getOrderBy());
        }
        
        // 添加$expand参数
        if (StrUtil.isNotBlank(query.getExpand())) {
            params.add("$expand=" + query.getExpand());
        }
        
        // 添加$top参数
        if (query.getTop() > 0) {
            params.add("$top=" + query.getTop());
        }
        
        // 注意: Dataverse不支持$skip参数（会返回错误0x80060888）
        // 分页必须通过@odata.nextLink实现
        // $skip参数已被移除
        
        // 添加$count参数
        if (query.isCount()) {
            params.add("$count=true");
        }
        
        // 拼接参数
        if (!params.isEmpty()) {
            url.append("?").append(String.join("&", params));
        }
        
        return url.toString();
    }
    
    /**
     * 处理HTTP错误响应
     * 解析OData错误消息并抛出适当的异常
     *
     * @param response HTTP响应
     * @param url 请求URL
     */
    private void handleError(HttpResponse response, String url) {
        int status = response.getStatus();
        String body = response.body();
        
        // 尝试解析OData错误响应
        String errorCode = null;
        String errorMessage = null;
        
        try {
            JSONObject json = JSON.parseObject(body);
            JSONObject error = json.getJSONObject("error");
            
            if (error != null) {
                errorCode = error.getString("code");
                errorMessage = error.getString("message");
            }
        } catch (Exception e) {
            log.debug("Failed to parse OData error response", e);
        }
        
        // 根据HTTP状态码抛出相应异常
        if (status == 401) {
            // 401 Unauthorized - 认证失败
            String message = errorMessage != null ? errorMessage : "Authentication failed";
            throw new DataverseAuthException("Dataverse authentication failed [{}]: {} (URL: {})", 
                    errorCode != null ? errorCode : "401", message, url);
        } else if (status == 403) {
            // 403 Forbidden - 权限不足
            String message = errorMessage != null ? errorMessage : "Access forbidden";
            throw new DataverseAuthException("Dataverse access forbidden [{}]: {} (URL: {})", 
                    errorCode != null ? errorCode : "403", message, url);
        } else if (status == 404) {
            // 404 Not Found - 实体不存在
            String message = errorMessage != null ? errorMessage : "Entity not found";
            // 尝试从URL提取实体名称
            String entityName = extractEntityName(url);
            throw new EntityNotFoundException(entityName, url);
        } else if (status >= 400 && status < 500) {
            // 其他4xx错误 - 客户端错误
            String message = errorMessage != null ? errorMessage : "Bad request";
            throw new DataverseQueryException(
                    String.format("Dataverse query error [%s]: %s", 
                            errorCode != null ? errorCode : String.valueOf(status), message), 
                    url);
        } else if (status >= 500) {
            // 5xx错误 - 服务器错误
            String message = errorMessage != null ? errorMessage : "Server error";
            throw new DataverseConnectionException(
                    String.format("Dataverse server error [%s]: %s (URL: {})", 
                            errorCode != null ? errorCode : String.valueOf(status), message), 
                    url);
        } else {
            // 其他错误
            String message = errorMessage != null ? errorMessage : "Request failed";
            throw new DataverseQueryException(
                    String.format("Dataverse error [%s]: %s", 
                            errorCode != null ? errorCode : String.valueOf(status), message), 
                    url);
        }
    }
    
    /**
     * 从URL中提取实体名称
     *
     * @param url 请求URL
     * @return 实体名称
     */
    private String extractEntityName(String url) {
        try {
            // URL格式: https://org.crm.dynamics.com/api/data/v9.2/accounts?...
            // 或: https://org.crm.dynamics.com/api/data/v9.2/accounts(id)
            String path = url.substring(baseUrl.length());
            int queryIndex = path.indexOf('?');
            int parenIndex = path.indexOf('(');
            
            if (queryIndex > 0) {
                path = path.substring(0, queryIndex);
            }
            if (parenIndex > 0) {
                path = path.substring(0, parenIndex);
            }
            
            return path;
        } catch (Exception e) {
            return "unknown";
        }
    }
    
    /**
     * 解析OData响应
     * 提取value数组、nextLink和count
     * 同时处理格式化值注释
     *
     * @param body 响应体JSON字符串
     * @return ODataResponse对象
     */
    private ODataResponse parseResponse(String body) {
        try {
            log.debug("Parsing OData response: {}", body);
            JSONObject json = JSON.parseObject(body);
            ODataResponse response = new ODataResponse();
            
            // 提取value数组
            JSONArray value = json.getJSONArray("value");
            if (value != null) {
                response.setValue(value);
            } else {
                // 如果没有value数组，可能是单个实体响应
                // 将单个对象包装成数组
                JSONArray singleValue = new JSONArray();
                singleValue.add(json);
                response.setValue(singleValue);
            }
            
            // 提取nextLink用于分页
            String nextLink = json.getString("@odata.nextLink");
            response.setNextLink(nextLink);
            
            // 提取count
            Long count = json.getLong("@odata.count");
            response.setCount(count);
            
            log.debug("Parsed OData response: {} records, nextLink={}, count={}", 
                    response.getValue() != null ? response.getValue().size() : 0,
                    nextLink != null ? "present" : "null",
                    count);
            
            return response;
            
        } catch (Exception e) {
            log.error("Failed to parse OData response", e);
            throw new LambdaQueryException("Failed to parse OData response", e);
        }
    }
}
