package com.github.xuejike.query.dataverse.config;

import lombok.Data;

import java.util.HashMap;
import java.util.Map;

/**
 * Configuration for Dataverse connections.
 * Supports multiple environments with different authentication settings.
 *
 * @author xuejike
 */
@Data
public class DataverseConfig {
    
    /**
     * Map of environment name to environment configuration
     */
    private Map<String, DataverseEnvironment> environments = new HashMap<>();
    
    /**
     * Token cache configuration
     */
    private TokenCacheConfig tokenCache = new TokenCacheConfig();
    
    /**
     * HTTP client configuration
     */
    private HttpConfig http = new HttpConfig();
    
    /**
     * Get the environment configuration by name.
     *
     * @param environmentName the environment name
     * @return the environment configuration, or null if not found
     */
    public DataverseEnvironment getEnvironment(String environmentName) {
        return environments.get(environmentName);
    }
    
    /**
     * Add an environment configuration.
     *
     * @param name the environment name
     * @param environment the environment configuration
     */
    public void addEnvironment(String name, DataverseEnvironment environment) {
        environments.put(name, environment);
    }
    
    /**
     * Token cache configuration.
     */
    @Data
    public static class TokenCacheConfig {
        /**
         * Whether token caching is enabled
         */
        private boolean enabled = true;
        
        /**
         * Token time-to-live in seconds
         */
        private long ttl = 3600;
    }
    
    /**
     * HTTP client configuration.
     */
    @Data
    public static class HttpConfig {
        /**
         * Connection timeout in milliseconds
         */
        private int connectTimeout = 30000;
        
        /**
         * Read timeout in milliseconds
         */
        private int readTimeout = 60000;
        
        /**
         * Maximum number of retries
         */
        private int maxRetries = 3;
    }
}
