package com.github.xuejike.query.dataverse.metadata;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.github.xuejike.query.dataverse.client.DataverseClient;
import com.github.xuejike.query.dataverse.client.ODataResponse;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * Service for querying Dataverse metadata.
 * Provides methods to retrieve entity and field metadata with caching.
 *
 * @author xuejike
 */
@Slf4j
public class MetadataService {
    
    private final DataverseClient client;
    private final Map<String, EntityMetadata> metadataCache;
    
    /**
     * Create a new MetadataService.
     *
     * @param client the Dataverse client
     */
    public MetadataService(DataverseClient client) {
        this.client = client;
        this.metadataCache = new ConcurrentHashMap<>();
    }
    
    /**
     * Get entity metadata by logical name.
     * Results are cached for performance.
     *
     * @param entityLogicalName the entity logical name
     * @return the entity metadata
     */
    public EntityMetadata getEntityMetadata(String entityLogicalName) {
        return metadataCache.computeIfAbsent(entityLogicalName, this::fetchEntityMetadata);
    }
    
    /**
     * List all available entities in the Dataverse environment.
     *
     * @return list of entity logical names
     */
    public List<String> listAllEntities() {
        log.debug("Fetching list of all entities");
        
        // 查询EntityDefinitions，只选择LogicalName和DisplayName字段
        String url = "EntityDefinitions?$select=LogicalName,DisplayName";
        ODataResponse response = client.get(url);
        
        // 提取实体逻辑名称列表
        List<String> entities = new ArrayList<>();
        JSONArray value = response.getValue();
        
        if (value != null) {
            for (int i = 0; i < value.size(); i++) {
                JSONObject item = value.getJSONObject(i);
                String logicalName = item.getString("LogicalName");
                if (logicalName != null) {
                    entities.add(logicalName);
                }
            }
        }
        
        log.debug("Found {} entities", entities.size());
        return entities;
    }
    
    /**
     * Clear the metadata cache.
     */
    public void clearCache() {
        metadataCache.clear();
        log.debug("Metadata cache cleared");
    }
    
    /**
     * Clear cached metadata for a specific entity.
     *
     * @param entityLogicalName the entity logical name
     */
    public void clearCache(String entityLogicalName) {
        metadataCache.remove(entityLogicalName);
        log.debug("Cleared metadata cache for entity: {}", entityLogicalName);
    }
    
    /**
     * 从Dataverse获取实体元数据
     * 查询EntityDefinitions API并解析响应
     *
     * @param entityLogicalName 实体逻辑名称
     * @return 实体元数据
     */
    private EntityMetadata fetchEntityMetadata(String entityLogicalName) {
        log.debug("Fetching metadata for entity: {}", entityLogicalName);
        
        // 构建元数据查询URL
        // 使用$expand=Attributes来获取所有字段信息
        String url = String.format(
            "EntityDefinitions(LogicalName='%s')?$expand=Attributes", 
            entityLogicalName);
        
        ODataResponse response = client.get(url);
        JSONArray value = response.getValue();
        
        if (value == null || value.isEmpty()) {
            log.warn("No metadata found for entity: {}", entityLogicalName);
            return null;
        }
        
        // 解析实体元数据
        JSONObject json = value.getJSONObject(0);
        EntityMetadata metadata = new EntityMetadata();
        
        // 设置基本信息
        metadata.setLogicalName(json.getString("LogicalName"));
        
        // 解析DisplayName（可能是LocalizedLabel对象）
        String displayName = extractDisplayName(json.getJSONObject("DisplayName"));
        metadata.setDisplayName(displayName);
        
        // 设置主键字段名
        metadata.setPrimaryIdAttribute(json.getString("PrimaryIdAttribute"));
        metadata.setPrimaryNameAttribute(json.getString("PrimaryNameAttribute"));
        
        // 解析字段元数据
        JSONArray attributes = json.getJSONArray("Attributes");
        if (attributes != null) {
            List<FieldMetadata> fields = new ArrayList<>();
            for (int i = 0; i < attributes.size(); i++) {
                JSONObject attr = attributes.getJSONObject(i);
                FieldMetadata field = parseFieldMetadata(attr);
                fields.add(field);
            }
            metadata.setFields(fields);
            log.debug("Parsed {} fields for entity {}", fields.size(), entityLogicalName);
        }
        
        return metadata;
    }
    
    /**
     * 从LocalizedLabel对象中提取显示名称
     *
     * @param displayNameObj DisplayName JSON对象
     * @return 显示名称字符串
     */
    private String extractDisplayName(JSONObject displayNameObj) {
        if (displayNameObj == null) {
            return null;
        }
        
        // DisplayName可能包含UserLocalizedLabel对象
        JSONObject userLabel = displayNameObj.getJSONObject("UserLocalizedLabel");
        if (userLabel != null) {
            String label = userLabel.getString("Label");
            if (label != null) {
                return label;
            }
        }
        
        // 或者直接包含LocalizedLabels数组
        JSONArray labels = displayNameObj.getJSONArray("LocalizedLabels");
        if (labels != null && !labels.isEmpty()) {
            JSONObject firstLabel = labels.getJSONObject(0);
            return firstLabel.getString("Label");
        }
        
        return null;
    }
    
    /**
     * 解析字段元数据
     * 根据字段类型提取相应的约束信息
     *
     * @param attr 字段属性JSON对象
     * @return 字段元数据
     */
    private FieldMetadata parseFieldMetadata(JSONObject attr) {
        FieldMetadata field = new FieldMetadata();
        
        // 基本信息
        field.setLogicalName(attr.getString("LogicalName"));
        
        // 解析DisplayName
        String displayName = extractDisplayName(attr.getJSONObject("DisplayName"));
        field.setDisplayName(displayName);
        
        // 字段类型
        String attributeType = attr.getString("AttributeType");
        field.setAttributeType(attributeType);
        
        // 是否必填
        // RequiredLevel可能是对象或字符串
        Object requiredLevel = attr.get("RequiredLevel");
        boolean isRequired = false;
        if (requiredLevel != null) {
            if (requiredLevel instanceof JSONObject) {
                String value = ((JSONObject) requiredLevel).getString("Value");
                isRequired = "ApplicationRequired".equals(value) || "SystemRequired".equals(value);
            } else if (requiredLevel instanceof String) {
                isRequired = "ApplicationRequired".equals(requiredLevel) || "SystemRequired".equals(requiredLevel);
            }
        }
        field.setRequired(isRequired);
        
        // 是否为主键
        Boolean isPrimaryId = attr.getBoolean("IsPrimaryId");
        field.setPrimaryKey(isPrimaryId != null && isPrimaryId);
        
        // 根据字段类型解析特定约束
        if ("String".equals(attributeType) || "Memo".equals(attributeType)) {
            // 字符串类型：最大长度
            Integer maxLength = attr.getInteger("MaxLength");
            field.setMaxLength(maxLength);
            
        } else if ("Decimal".equals(attributeType) || "Money".equals(attributeType)) {
            // 数值类型：精度和小数位数
            Integer precision = attr.getInteger("Precision");
            field.setPrecision(precision);
            
            // Money类型的精度信息可能在PrecisionSource中
            if (precision == null) {
                precision = attr.getInteger("PrecisionSource");
                field.setPrecision(precision);
            }
            
            // 最小值和最大值
            Double minValue = attr.getDouble("MinValue");
            Double maxValue = attr.getDouble("MaxValue");
            field.setMinValue(minValue);
            field.setMaxValue(maxValue);
            
        } else if ("Integer".equals(attributeType)) {
            // 整数类型：最小值和最大值
            Double minValue = attr.getDouble("MinValue");
            Double maxValue = attr.getDouble("MaxValue");
            field.setMinValue(minValue);
            field.setMaxValue(maxValue);
            
        } else if ("Double".equals(attributeType)) {
            // 双精度浮点数：精度、最小值和最大值
            Integer precision = attr.getInteger("Precision");
            field.setPrecision(precision);
            
            Double minValue = attr.getDouble("MinValue");
            Double maxValue = attr.getDouble("MaxValue");
            field.setMinValue(minValue);
            field.setMaxValue(maxValue);
            
        } else if ("Lookup".equals(attributeType) || "Customer".equals(attributeType) || "Owner".equals(attributeType)) {
            // 查找字段：目标实体
            JSONArray targets = attr.getJSONArray("Targets");
            if (targets != null && !targets.isEmpty()) {
                List<String> lookupTargets = new ArrayList<>();
                for (int i = 0; i < targets.size(); i++) {
                    String target = targets.getString(i);
                    if (target != null) {
                        lookupTargets.add(target);
                    }
                }
                field.setLookupTargets(lookupTargets);
            }
        }
        
        return field;
    }
}
