package cdc.kernel.validation;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.jupiter.api.Test;

import cdc.validation.checkers.Checker;
import cdc.validation.checkers.defaults.HasNoDoubleSpaces;
import cdc.validation.checkers.defaults.HasNoOuterSpaces;
import cdc.validation.checkers.defaults.HasNoOuterWhiteSpaces;
import cdc.validation.checkers.defaults.HasNoSpaces;
import cdc.validation.checkers.defaults.HasNoWhiteSpaces;
import cdc.validation.checkers.defaults.IsInRange;
import cdc.validation.checkers.defaults.IsInstanceOf;
import cdc.validation.checkers.defaults.IsNotNull;

class CheckersDefaultTest {
    private static final Logger LOGGER = LogManager.getLogger(CheckersDefaultTest.class);

    @Test
    void testIsNotNull() {
        assertEquals(false, IsNotNull.INSTANCE.test(null));
        assertEquals(true, IsNotNull.INSTANCE.test("Hello"));
        assertEquals(true, IsNotNull.INSTANCE.negate().test(null));
        assertEquals(false, IsNotNull.INSTANCE.negate().test("Hello"));
        LOGGER.debug(IsNotNull.INSTANCE.explain());
        LOGGER.debug(IsNotNull.INSTANCE.negate().explain());
        LOGGER.debug(IsNotNull.INSTANCE.testAndExplain(null));
        LOGGER.debug(IsNotNull.INSTANCE.testAndExplain("Hello"));
        LOGGER.debug(IsNotNull.INSTANCE.negate().testAndExplain(null));
        LOGGER.debug(IsNotNull.INSTANCE.negate().testAndExplain("Hello"));
    }

    @Test
    void testIsInRange() {
        assertEquals(true, IsInRange.from(0, 100).test(10));
        assertEquals(false, IsInRange.from(0, 100).test(200));
        LOGGER.debug(IsInRange.from(0, 100).explain());
        LOGGER.debug(IsInRange.from(0, 100).negate().explain());
        LOGGER.debug(IsInRange.from(0, 100).testAndExplain(10));
        LOGGER.debug(IsInRange.from(0, 100).testAndExplain(200));
        LOGGER.debug(IsInRange.from(0, 100).negate().testAndExplain(10));
        LOGGER.debug(IsInRange.from(0, 100).negate().testAndExplain(200));
    }

    @Test
    void testCharacter() {
        assertTrue(Character.isSpaceChar(' '));
        assertFalse(Character.isSpaceChar('\n'));
        assertFalse(Character.isSpaceChar('\t'));

        assertTrue(Character.isWhitespace(' '));
        assertTrue(Character.isWhitespace('\n'));
        assertTrue(Character.isWhitespace('\n'));
    }

    @Test
    void testIsInstanceOf() {
        final Checker<Object> checker = IsInstanceOf.from(String.class);
        assertEquals(false, checker.test(null));
        assertEquals(true, checker.test("Hello"));
        LOGGER.debug(checker.explain());
    }

    @Test
    void testHasNoSpaces() {
        final Checker<String> checker = HasNoSpaces.INSTANCE;
        assertTrue(checker.test("a"));
        assertFalse(checker.test("a a"));
        assertTrue(checker.test("a\ta"));
        assertTrue(checker.test("a\na"));
    }

    @Test
    void testHasNoWhiteSpaces() {
        final Checker<String> checker = HasNoWhiteSpaces.INSTANCE;
        assertTrue(checker.test("a"));
        assertFalse(checker.test("a a"));
        assertFalse(checker.test("a\ta"));
        assertFalse(checker.test("a\na"));
    }

    @Test
    void testHasNoOuterSpaces() {
        final Checker<String> checker = HasNoOuterSpaces.INSTANCE;
        assertTrue(checker.test("a"));
        assertTrue(checker.test("a a"));
        assertTrue(checker.test("a\ta"));
        assertTrue(checker.test("a\na"));

        assertFalse(checker.test("a "));
        assertTrue(checker.test("a\n"));
        assertTrue(checker.test("a\t"));

        assertFalse(checker.test(" a"));
        assertTrue(checker.test("\na"));
        assertTrue(checker.test("\ta"));
    }

    @Test
    void testHasNoOuterWhiteSpaces() {
        final Checker<String> checker = HasNoOuterWhiteSpaces.INSTANCE;
        assertTrue(checker.test("a"));
        assertTrue(checker.test("a a"));
        assertTrue(checker.test("a\ta"));
        assertTrue(checker.test("a\na"));

        assertFalse(checker.test("a "));
        assertFalse(checker.test("a\t"));
        assertFalse(checker.test("a\n"));

        assertFalse(checker.test(" a"));
        assertFalse(checker.test("\ta"));
        assertFalse(checker.test("\na"));
    }

    @Test
    void testHasNoDoubleSpaces() {
        final Checker<String> checker = HasNoDoubleSpaces.INSTANCE;
        assertTrue(checker.test("a"));
        assertTrue(checker.test("a "));
        assertFalse(checker.test("a  "));
        assertTrue(checker.test("a \n"));
        assertTrue(checker.test("a\n "));
    }
}