package com.googlecode.java_cl_parser;

import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Stack;

public class StringUtil
{
	private static final String PUNCTUATION = ".;:'\",`?/\\~!@#$%^&*()-=_+{}[]|<>";

	public static List<String> wrapTextToList(String text, int width)
	{
		if (width <= 1)
		{
			throw new IllegalArgumentException("Less than two-width is not supported");
		}

		List<String> list = new ArrayList<String>();

		List<String> wordList = splitWords(text);

		Stack<String> wordStack = new Stack<String>();

		// reverse order so item [0] comes out of the stack first
		Collections.reverse(wordList);

		wordStack.addAll(wordList);

		StringBuilder build = new StringBuilder();

		while (!wordStack.empty())
		{
			// add next word to end of line
			if (build.length() != 0)
			{
				build.append(' ');
			}

			build.append(wordStack.pop());

			// check if we have exceeded the acceptable length
			if (build.length() == width)
			{
				// right on - push it and clear the line
				list.add(build.toString());
				build.setLength(0);
			}
			else if (build.length() > width)
			{
				// we went too far - append the acceptable length to the list and push the rest back onto the stack
				String fragment = null;

				// check for a few special cases here.
				// is the last character a space?  If so, simply leave it.
				if (build.charAt(width - 1)  == ' ')
				{
					fragment = build.substring(0, width);

					build.delete(0, width);
				}
				// is the second-to-last character a space?  If so, make the last character a space and push the new character to
				// the next line.
				else if (build.charAt(width - 2)  == ' ')
				{
					fragment = build.substring(0, width - 1) + ' ';

					build.delete(0, width - 1);
				}
				// is the last character a punctuation mark? If so, simply leave it.
				else if (PUNCTUATION.indexOf(build.charAt(width - 1)) != -1)
				{
					fragment = build.substring(0, width);

					build.delete(0, width);
				}
				else
				{
					fragment = build.substring(0, width - 1) + '-';

					build.delete(0, width - 1);

					// push it back onto the stack so that it can be processed next, and clear the builder
					wordStack.push(build.toString());
					build.setLength(0);
				}

				list.add(fragment);
			}
		}

		if (build.length() != 0 || list.size() == 0)
		{
			while (build.length() < width)
			{
				build.append(' ');
			}

			list.add(build.toString());
		}

		return list;
	}

	/**
	 * Does two things - strips all non-printable characters and splits on white space
	 * @param text
	 */
	public static List<String> splitWords(String text)
	{
		List<String> parts = new ArrayList<String>();

		StringBuilder stb = new StringBuilder();

		for (int offset = 0; offset < text.length(); offset++)
		{
			char ch = text.charAt(offset);

			boolean javaIdentifierPart = Character.isJavaIdentifierPart(ch);
			boolean whitespace = Character.isWhitespace(ch);
			boolean punctuation = PUNCTUATION.indexOf(ch) != -1;

			// ignore if not in our handled group
			if (javaIdentifierPart || whitespace || punctuation)
			{
				if (!whitespace)
				{
					stb.append(ch);
				}
				else
				{
					if (stb.length() != 0)
					{
						parts.add(stb.toString());
						stb.setLength(0);
					}
				}
			}
		}

		if (stb.length() != 0)
		{
			parts.add(stb.toString());
		}

		return parts;
	}
}
