package com.googlecode.jpattern.core.textfiles;

import java.util.Date;

import com.googlecode.jpattern.core.command.ACommand;
import com.googlecode.jpattern.core.command.ICommand;
import com.googlecode.jpattern.core.command.ICommandResult;
import com.googlecode.jpattern.core.command.NullCommand;
import com.googlecode.jpattern.service.log.ILogger;
import com.googlecode.jpattern.shared.result.ErrorMessage;

/**
 * 
 * @author Francesco Cina'
 *
 * 10/giu/2010
 */
public class FileCopyCommand extends ACommand {

	private static final long serialVersionUID = 1L;
	private ILogger logger;
	private StringBuffer sourceFileName;
	private IResource source;
	private IResource destination;
	private StringBuffer destinationFileName;
	
	public FileCopyCommand(StringBuffer sourceFileName, IResource source, StringBuffer destinationFileName, IResource destination) {
		this( sourceFileName, source, destinationFileName, destination, new NullCommand() );
	}
	
	public FileCopyCommand(StringBuffer sourceFileName, IResource source, StringBuffer destinationFileName, IResource destination, ICommand aSuccessor) {
		super(aSuccessor);
		this.sourceFileName = sourceFileName;
		this.source = source;
		this.destination = destination;
		this.destinationFileName = destinationFileName;
	}
	
	@Override
	protected void internalRollBack(ICommandResult result) {
	}

	@Override
	protected void result(ICommandResult result) {
		logger = getProvider().getLoggerService().logger(FileCopyCommand.class);
		logger.info("result", "copy file " + sourceFileName.toString() + " from " + source.getPath() + " to " + destinationFileName.toString() + " in " + destination.getPath() );
		
		if ( !source.getFilenames().contains(sourceFileName.toString()) ) {
			logger.error("result", "Impossible to copy file: source file doesn't exists");
			result.getErrorMessages().add( new ErrorMessage("FileCopyCommand", "Impossible to copy file: source file doesn't exists") );
			return;
		}
		
		if ( destination.getFilenames().contains(destinationFileName.toString()) ) {
			logger.error("result", "Impossible to copy file: destination file with same name exists");
			result.getErrorMessages().add( new ErrorMessage("FileCopyCommand", "Impossible to copy file: destination file with same exists") );
			return;
		}
		
		FileWrapper destinationWrapper = new FileWrapper();
		StringBuffer temporaryName = new StringBuffer( destinationFileName.toString() );
		long now = new Date().getTime();
		temporaryName.append( "_temp_" + now );
		ICommand command = new FileCreateCommand(destination, temporaryName, new StringBuffer(), destinationWrapper);
		command.visit( getProvider() );
		ICommandResult createResult = command.exec();
		
		if ( createResult.isValid() ) {
			IFileReader sourceFileReader = null;
			IFileWriter destinationFileWriter = null;
			try {
				IFile sourceFile = source.getFile( sourceFileName.toString() );
				sourceFileReader = sourceFile.getFileReader();
				destinationFileWriter = destinationWrapper.getFile().getFileWriter(false);
				while ( sourceFileReader.hasNextLine() ) {
					destinationFileWriter.println( sourceFileReader.readLine() );
				}
				sourceFileReader.close();
				destinationFileWriter.close();
				
				command = new FileRenameCommand(temporaryName, destinationFileName, destination);
				command.visit(getProvider());
				ICommandResult renameResult = command.exec();
				
				if ( !renameResult.isValid() ) {
					logger.error("result", "error writing file in destination " + destination.getPath());
					result.getErrorMessages().addAll( createResult.getErrorMessages() );
				}
			}
			catch (Exception e) {
				logger.error("result", "error writing file in destination " + destination.getPath(), e);
			}
			finally {
				if (destinationFileWriter!=null) destinationFileWriter.close();
				if (sourceFileReader!=null) sourceFileReader.close();
			}
		}
		else {
			result.getErrorMessages().addAll( createResult.getErrorMessages() );
		}
		
	}

}
