package com.jpattern.orm.mapper;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.jpattern.logger.ILogger;
import com.jpattern.orm.logger.OrmLogger;
import com.jpattern.orm.util.FieldDefaultNaming;

/**
 * 
 * @author Francesco Cina
 *
 * 22/mag/2011
 */
public class ClassMapperBuilder<T> implements IClassMapperBuilder<T> {

	private final Class<T> clazz;
	private ILogger logger = OrmLogger.getOrmLogger(this.getClass());
	private final ITableMap tableMap;

	public ClassMapperBuilder(Class<T> clazz, ITableMap tableMap) {
		this.clazz = clazz;
		this.tableMap = tableMap;
	}

	/* (non-Javadoc)
	 * @see com.jpattern.orm.mapper.IClassMapperGenerator#generate()
	 */
	@Override
	public IClassMapper<T> generate() {
		logger.info("generate", "generate " + IClassMapper.class.getSimpleName() + " for Class " + clazz.getName());
		String tableName = tableMap.getTableNameWithSchema();
		String schemaName = tableMap.getSchemaName();
		logger.info("generate", "table name expected in relation with class " + clazz.getSimpleName() + ": " + tableName + " - schema: " + schemaName);
		Map<String, IColumn> fieldColumnMapping = mapField(schemaName, tableName);
		ClassMapper<T> classMapper = new ClassMapper<T>(clazz, tableMap, fieldColumnMapping);
		initializeColumnNames(classMapper, fieldColumnMapping);
		return classMapper ;
	}

	private Map<String, IColumn> mapField(String schemaName, String tableName) {
		Map<String, IColumn> result = new LinkedHashMap<String, IColumn>();
		Column col = null;
		
		for ( String fieldName : tableMap.getAllFieldNames() ) {
			if ( fieldName.length() > 0) {
				String columnName = tableMap.getDBColumnName(fieldName);
				logger.info("exploreAllColumns", "DB column [" + columnName + "]" + " will be associated with object field [" + fieldName + "]");
				col = new Column( columnName );
				col.setIdentifying( tableMap.getPrimaryKeyFieldNames().contains(fieldName) );
				setGetterSetter( fieldName, col );
				result.put( fieldName , col );
			} else {
				logger.warn("exploreAllColumns", "Class fiels [" + fieldName + "]" + " is NOT associated with a DB colunm!");
			}
		}
		return result;
	}

    private void setGetterSetter(String javaPropertyName, Column col) {
		List<Method> methods = Arrays.asList( clazz.getMethods() );
		List<Field> fields = Arrays.asList( clazz.getDeclaredFields() );
		
		Field propertyField = null;
		Method getter = null;
		Method setter = null;
		String fieldName = "";
		String getterName = "";
		String setterName = "";

		for (Field field : fields) {
			if (javaPropertyName.equals(field.getName())) {
				propertyField = field;
				fieldName = field.getName();
			}
		}
		
		for (Method method : methods) {
			if (FieldDefaultNaming.getDefaultGetterName(javaPropertyName).equals(method.getName())) {
				getter = method;
				getterName = method.getName();
			}
			if (FieldDefaultNaming.getDefaultBooleanGetterName(javaPropertyName).equals(method.getName())) {
				getter = method;
				getterName = method.getName();
			}
			if (FieldDefaultNaming.getDefaultSetterName(javaPropertyName).equals(method.getName())) {
				setter = method;
				setterName = method.getName();
			}
		}
		
		logger.info("setGetterSetter", "property [" + fieldName + "] associated with column [" + col.getName() + "]");
		logger.info("setGetterSetter", "getter for property [" + fieldName + "]: [" + getterName + "]");
		logger.info("setGetterSetter", "setter for property [" + fieldName + "]: [" + setterName + "]");
		
		col.setField(propertyField);
		col.setSetter(setter);
		col.setGetter(getter);
	}
    
	private void initializeColumnNames(ClassMapper<T> generated, Map<String, IColumn> fieldColumnMapping) {
		logger.info("initializeColumnNames", "");
		List<String> allColumnJavaNamesList = new ArrayList<String>();
		List<String> allNotGeneratedColumnJavaNamesList = new ArrayList<String>();
		List<String> allGeneratedColumnJavaNamesList = new ArrayList<String>();
		List<String> allGeneratedColumnDBNamesList = new ArrayList<String>();
		List<String> primaryKeyColumnJavaNamesList = new ArrayList<String>();
		List<String> notPrimaryKeyColumnJavaList = new ArrayList<String>();
		for (Entry<String, IColumn> entry : fieldColumnMapping.entrySet()) {
			String javaFieldName = entry.getKey();
			allColumnJavaNamesList.add(javaFieldName);
			if( entry.getValue().isIdentifying() ) {
				primaryKeyColumnJavaNamesList.add(javaFieldName);
				logger.info("initializeColumnNames", "Field [" + javaFieldName + "] will be used as a Primary Key field");
			} else {
				notPrimaryKeyColumnJavaList.add(javaFieldName);
				logger.info("initializeColumnNames", "Field [" + javaFieldName + "] will be used as a normal field");
			}
			if (tableMap.getGeneratorByJavaFieldName(javaFieldName).isAutoGenerated()) {
				allGeneratedColumnJavaNamesList.add(javaFieldName);
				allGeneratedColumnDBNamesList.add(entry.getValue().getName());
				logger.info("initializeColumnNames", "Field [" + javaFieldName + "] is an autogenerated field");
			} else {
				allNotGeneratedColumnJavaNamesList.add(javaFieldName);
			}
			
		}
		
		generated.setAllColumnJavaNames( allColumnJavaNamesList.toArray(new String[0]) );
		generated.setAllNotGeneratedColumnJavaNames( allNotGeneratedColumnJavaNamesList.toArray(new String[0]) );
		generated.setAllGeneratedColumnJavaNames( allGeneratedColumnJavaNamesList.toArray(new String[0]) );
		generated.setAllGeneratedColumnDBNames( allGeneratedColumnDBNamesList.toArray(new String[0]) );
		generated.setNotPrimaryKeyColumnJavaNames( notPrimaryKeyColumnJavaList.toArray(new String[0]) );
		generated.setPrimaryKeyColumnJavaNames( primaryKeyColumnJavaNamesList.toArray(new String[0]) );
		generated.setNotPrimaryKeyColumnJavaNames( notPrimaryKeyColumnJavaList.toArray(new String[0]) );
	}
}
