package com.jpattern.orm.query;

import java.util.Collection;
import java.util.Map;

/**
 * 
 * @author Francesco Cina
 * 
 *         18/giu/2011
 */
public interface IExpression extends IExpressionElement {
	
	/**
	 * All Equal - Map containing property names and their values.
	 * @param propertyMap
	 * @return
	 */
	IExpression allEq(Map<String,Object> propertyMap);

	/**
	 * Express the "Equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	IExpression eq(String property, Object value);
	
	/**
	 * Express the "Equals to" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	IExpression eqProperties(String firstProperty, String secondProperty);

	/**
	 * Express the "Lesser or equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	IExpression le(String property, Object value);
	
	/**
	 * Express the "Lesser or equals to" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	IExpression leProperties(String firstProperty, String secondProperty);

	/**
	 * Express the "Greater or equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	IExpression ge(String property, Object value);
	
	/**
	 * Express the "Greater or equals to" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	IExpression geProperties(String firstProperty, String secondProperty);

	/**
	 * 
	 * Express the "Lesser than" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	IExpression lt(String property, Object value);
	
	/**
	 * Express the "Lesser than" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	IExpression ltProperties(String firstProperty, String secondProperty);

	/**
	 * Express the "Greater than" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	IExpression gt(String property, Object value);
	
	/**
	 * Express the "Greater than" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	IExpression gtProperties(String firstProperty, String secondProperty);

	/**
	 * Express the "Insensitive Equal To" between an object's property
	 * and a fixed value (it uses a lower() function to make both case insensitive).
	 * 
	 * @param propertyName
	 * @param value
	 * @return
	 */
	IExpression ieq(String property, String value);
	
	/**
	 * Express the "Insensitive Equal To" bbetween objects properties
	 * (it uses a lower() function to make both case insensitive).
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	IExpression ieqProperties(String firstProperty, String secondProperty);

	/**
	 * Express the "Not Equals to" relation between objects properties.
	 *  
	 * @param property
	 * @param value
	 * @return
	 */
	IExpression ne(String property, Object value);
	
	/**
	 * Express the "Not Equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	IExpression neProperties(String firstProperty, String secondProperty);
	
	/**
	 * Case insensitive Like - property like value where the value contains the
	 * SQL wild card characters % (percentage) and _ (underscore).
	 * 
	 * @param propertyName
	 * @param value
	 * @return
	 */
	IExpression ilike(String property, String value);

	/**
	 * Not In - property has a value in the collection of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	IExpression nin(String property, Collection<?> values);

	/**
	 * Not In - property has a value in the array of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	IExpression nin(String property, Object[] values);

	/**
	 * Not In - using a subQuery.
	 * 
	 * @param propertyName
	 * @param subQuery
	 * @return
	 */
	IExpression nin(String property, IQuery subQuery);

	
	/**
	 * In - property has a value in the collection of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	IExpression in(String property, Collection<?> values);

	/**
	 * In - property has a value in the array of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	IExpression in(String property, Object[] values);

	/**
	 * In - using a subQuery.
	 * 
	 * @param propertyName
	 * @param subQuery
	 * @return
	 */
	IExpression in(String property, IQuery subQuery);

	/**
	 * Is Not Null - property is not null.
	 * 
	 * @param propertyName
	 * @return
	 */
	IExpression isNotNull(String property);

	/**
	 * Is Null - property is null.
	 * 
	 * @param propertyName
	 * @return
	 */
	IExpression isNull(String property);

	/**
	 * Like - property like value where the value contains the SQL wild card
	 * characters % (percentage) and _ (underscore).
	 * 
	 * @param propertyName
	 * @param value
	 */
	IExpression like(String property, String value);

	/**
	 * Not Like - property like value where the value contains the SQL wild card
	 * characters % (percentage) and _ (underscore).
	 * 
	 * @param propertyName
	 * @param value
	 */
	IExpression nlike(String property, String value);
	
	/**
	 * Negate the expression (prefix it with NOT).
	 * 
	 * @param exp
	 * @return
	 */
	IExpression not(IExpressionElement expression);

	/**
	 * Or - join two expressions with a logical or.
	 * 
	 * @param expOne
	 * @param expTwo
	 * @return
	 */
	IExpression or(IExpressionElement expressionOne, IExpressionElement expressionTwo);

	/**
	 * And - join two expressions with a logical and.
	 * 
	 * @param expOne
	 * @param expTwo
	 * @return
	 */
	IExpression and(IExpressionElement expressionOne, IExpressionElement expressionTwo);

}
