package com.jpattern.orm.query;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import com.jpattern.orm.IOrmClassToolMap;
import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.exception.OrmNotUniqueResultException;
import com.jpattern.orm.session.IResultSetReader;
import com.jpattern.orm.session.ISessionSqlPerformer;
import com.jpattern.orm.session.ISqlPerformer;

/**
 * 
 * @author Francesco Cina
 *
 * 20/giu/2011
 */
public class OrmCustomQuery extends ABaseOrmQuery implements IOrmCustomQuery, INameSolverConsumer {

	private INameSolver nameSolver = new NullNameSolver();
	private final ISelectClause select;
	private final IOrmClassToolMap ormClassToolMap;
	private final Class<?> clazz;
	private final Class<?>[] joinClasses;
	private final ISessionSqlPerformer session;
	private int queryTimeout = 0;
	private int maxRows = 0;

	public OrmCustomQuery(String selectClause, IOrmClassToolMap ormClassToolMap, ISessionSqlPerformer session, Class<?> clazz, Class<?>... joinClasses) {
		this.ormClassToolMap = ormClassToolMap;
		this.session = session;
		this.clazz = clazz;
		this.joinClasses = joinClasses;
		setJoin( new Join(ormClassToolMap) );
		this.select = new SelectClause(selectClause);
	}

	@Override
	public void setNameSolver(INameSolver nameSolver) {
		this.nameSolver = nameSolver;
		this.where().setNameSolver(nameSolver);
		this.orderBy().setNameSolver(nameSolver);
		this.join().setNameSolver(nameSolver);
		this.select.setNameSolver(nameSolver);
	}

	@Override
	public List<Object[]> findList() {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForList(renderSql(), values.toArray());
	}


	@Override
	public Object[] findUnique() throws OrmNotUniqueResultException {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForArray(renderSql(), values.toArray());
	}

	@Override
	protected void renderSelect(StringBuilder StringBuilder) {
		select.renderSql(StringBuilder);
	}

	@Override
	protected void renderFrom(StringBuilder StringBuilder) {
		String alias = nameSolver.alias(clazz);
		StringBuilder.append("FROM ");
		StringBuilder.append(ormClassToolMap.getOrmClassTool(clazz).getClassMapper().getTableMap().getTableNameWithSchema() );
		StringBuilder.append( " " );
		StringBuilder.append(alias);
		StringBuilder.append(" ");
		join().renderSql(StringBuilder);
		if (joinClasses!=null && joinClasses.length>0) {
			for (Class<?> joinClass : joinClasses) {
				StringBuilder.append( ", " );
				StringBuilder.append(ormClassToolMap.getOrmClassTool(joinClass).getClassMapper().getTableMap().getTableNameWithSchema() );
				StringBuilder.append( " " );
				StringBuilder.append(nameSolver.alias(joinClass));
			}
			StringBuilder.append(" ");
		}
	}
	
	@Override
	protected void renderWhere(StringBuilder StringBuilder) {
		where().renderSql(StringBuilder);
	}
	
	@Override
	protected void renderOrderBy(StringBuilder StringBuilder) {
		orderBy().renderSql(StringBuilder);
	}

	@Override
	public <T> T find(IResultSetReader<T> rse) throws OrmException {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.query(renderSql(), rse, values.toArray());
	}

	@Override
	public int findInt() throws OrmException {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForInt(renderSql(), values.toArray());
	}

	@Override
	public long findLong() throws OrmException {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForLong(renderSql(), values.toArray());
	}

	@Override
	public double findDouble() throws OrmException {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForDouble(renderSql(), values.toArray());
	}

	@Override
	public float findFloat() throws OrmException {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForFloat(renderSql(), values.toArray());
	}

	@Override
	public String findString() throws OrmException {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForString(renderSql(), values.toArray());
	}

	@Override
	public boolean findBoolean() throws OrmException {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForBoolean(renderSql(), values.toArray());
	}

	@Override
	public BigDecimal findBigDecimal() throws OrmException {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForBigDecimal(renderSql(), values.toArray());
	}

	@Override
	public final int getMaxRows() throws OrmException {
		return maxRows;
	}

	@Override
	public final IOrmCustomQuery setQueryTimeout(int queryTimeout) {
		this.queryTimeout = queryTimeout;
		return this;
	}

	@Override
	public final int getQueryTimeout() {
		return queryTimeout;
	}

	@Override
	public final IOrmCustomQuery setMaxRows(int maxRows) throws OrmException {
		this.maxRows = maxRows;
		return this;
	}
}
