package com.jpattern.orm.query;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import com.jpattern.orm.IOrmClassTool;
import com.jpattern.orm.IOrmClassToolMap;
import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.exception.OrmNotUniqueResultException;
import com.jpattern.orm.session.IResultSetReader;
import com.jpattern.orm.session.ISessionSqlPerformer;
import com.jpattern.orm.session.ISqlPerformer;

/**
 * 
 * @author Francesco Cina
 *
 * 20/giu/2011
 */
public class OrmQuery<T> extends ABaseOrmQuery implements IOrmQuery<T>, INameSolverConsumer {

	private INameSolver nameSolver = new NullNameSolver();
	private final IOrmClassToolMap ormClassToolMap;
	private final Class<T> clazz;
	private final Class<?>[] joinClasses;
	private boolean distinct;
	private final ISessionSqlPerformer session;
	private int queryTimeout = 0;
	private int maxRows = 0;
	
	public OrmQuery(IOrmClassToolMap ormClassToolMap, ISessionSqlPerformer session, Class<T> clazz, Class<?>... joinClasses) {
		this.ormClassToolMap = ormClassToolMap;
		this.session = session;
		this.clazz = clazz;
		this.joinClasses = joinClasses;
		setJoin( new Join(ormClassToolMap) );
	}

	@Override
	public final int getMaxRows() throws OrmException {
		return maxRows;
	}

	@Override
	public final IOrmQuery<T> setQueryTimeout(int queryTimeout) {
		this.queryTimeout = queryTimeout;
		return this;
	}

	@Override
	public final int getQueryTimeout() {
		return queryTimeout;
	}

	@Override
	public final IOrmQuery<T> setMaxRows(int maxRows) throws OrmException {
		this.maxRows = maxRows;
		return this;
	}
	
	@Override
	public void setNameSolver(INameSolver nameSolver) {
		this.nameSolver = nameSolver;
		where().setNameSolver(nameSolver);
		orderBy().setNameSolver(nameSolver);
		join().setNameSolver(nameSolver);
	}

	@Override
	public List<T> findList() {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		final IOrmClassTool<T> ormClassTool = ormClassToolMap.getOrmClassTool(clazz);
		
		IResultSetReader<List<T>> resultSetReader = new IResultSetReader<List<T>>() {
			@Override
			public List<T> read(ResultSet resultSet) throws SQLException { 
				List<T> resultList = new ArrayList<T>();
				int rowCount = 0;
				while ( resultSet.next() ) {
					resultList.add( ormClassTool.getOrmPersistor().mapRow("", resultSet, rowCount++) );
				}
				return resultList;
			}
		};
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.query(renderSql(), resultSetReader, values.toArray());
	}
	
	@Override
	public void find(final IOrmSerialResultReader<T> srr) throws OrmException {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		final IOrmClassTool<T> ormClassTool = ormClassToolMap.getOrmClassTool(clazz);
		
		IResultSetReader<Object> resultSetReader = new IResultSetReader<Object>() {
			@Override
			public Object read(ResultSet resultSet) throws SQLException { 
				int rowCount = 0;
				while ( resultSet.next() ) {
					srr.read( ormClassTool.getOrmPersistor().mapRow("", resultSet, rowCount) , rowCount );
					rowCount++;
				}
				return null;
			}
		};
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		sqlExec.query(renderSql(), resultSetReader, values.toArray());
	}

	@Override
	public long findRowCount() {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForLong(getGeneratedRowCountSql(), values.toArray());
	}

	@Override
	public T findUnique() throws OrmNotUniqueResultException {
		int oldMaxRow = getMaxRows();
		setMaxRows(2);
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		List<T> result = findList();
		setMaxRows(oldMaxRow);
		if (result.size()==0) {
			return null;
		}
		if (result.size()>1) {
			throw new OrmNotUniqueResultException("The query execution returned a number of rows higher than 1");
		}
		return result.get(0);
	}

	@Override
	public String getGeneratedRowCountSql() {
		StringBuilder StringBuilder = new StringBuilder();
		renderSelectRowCount(StringBuilder);
		renderFrom(StringBuilder);
		renderWhere(StringBuilder);
		return StringBuilder.toString();
	}

	@Override
	public IOrmQuery<T> setDistinct() {
		this.distinct = true;
		return this;
	}

	private void renderSelectRowCount(StringBuilder StringBuilder) {
		StringBuilder.append("SELECT COUNT(*) ");
	}
	
	@Override
	protected void renderSelect(StringBuilder StringBuilder) {
		String alias = nameSolver.alias(clazz); 
		StringBuilder.append("SELECT ");
		if (distinct) {
			StringBuilder.append("DISTINCT ");
		}
		StringBuilder.append(ormClassToolMap.getOrmClassTool(clazz).getOrmCRUDQuery().getBaseSelectClause(alias + ".") );
		StringBuilder.append(" ");
	}

	@Override
	protected void renderFrom(StringBuilder StringBuilder) {
		String alias = nameSolver.alias(clazz);
		StringBuilder.append("FROM ");
		StringBuilder.append(ormClassToolMap.getOrmClassTool(clazz).getClassMapper().getTableMap().getTableNameWithSchema() );
		StringBuilder.append( " " );
		StringBuilder.append(alias);
		StringBuilder.append(" ");
		join().renderSql(StringBuilder);
		if (joinClasses!=null && joinClasses.length>0) {
			for (Class<?> joinClass : joinClasses) {
				StringBuilder.append( ", " );
				StringBuilder.append(ormClassToolMap.getOrmClassTool(joinClass).getClassMapper().getTableMap().getTableNameWithSchema() );
				StringBuilder.append( " " );
				StringBuilder.append(nameSolver.alias(joinClass));
			}
			StringBuilder.append(" ");
		}
	}
	
	@Override
	protected void renderWhere(StringBuilder StringBuilder) {
		where().renderSql(StringBuilder);
	}
	
	@Override
	protected void renderOrderBy(StringBuilder StringBuilder) {
		orderBy().renderSql(StringBuilder);
	}

	@Override
	public boolean isDistinct() throws OrmException {
		return distinct;
	}

}
