package com.jpattern.orm.session;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import com.jpattern.orm.IOrmClassTool;
import com.jpattern.orm.IOrmClassToolMap;
import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.query.IDelete;
import com.jpattern.orm.query.ISqlExecutor;
import com.jpattern.orm.query.IUpdate;
import com.jpattern.orm.query.OrmCustomQuery;
import com.jpattern.orm.query.IOrmCustomQuery;
import com.jpattern.orm.query.IOrmQuery;
import com.jpattern.orm.query.OrmClassToolMapNameSolver;
import com.jpattern.orm.query.OrmDelete;
import com.jpattern.orm.query.OrmQuery;
import com.jpattern.orm.query.OrmUpdate;
import com.jpattern.orm.query.SqlExecutor;
import com.jpattern.orm.script.IScriptExecutor;
import com.jpattern.orm.script.ScriptExecutor;

/**
 * 
 * @author Francesco Cina
 *
 * 27/giu/2011
 */
public abstract class ASession implements ISessionSqlPerformer {

	private final IOrmClassToolMap ormClassToolMap;
	
	public ASession(IOrmClassToolMap ormClassToolMap) {
		this.ormClassToolMap = ormClassToolMap;
	}

	@Override
	public final <T> IOrmQuery<T> findQuery(Class<T> clazz, Class<?>... joinClasses) throws OrmException {
		OrmQuery<T> query = new OrmQuery<T>(ormClassToolMap , this, clazz, joinClasses);
		OrmClassToolMapNameSolver nameSolver = new OrmClassToolMapNameSolver(ormClassToolMap);
		nameSolver.register(clazz);
		if (joinClasses!=null) {
			for (Class<?> joinClass : joinClasses) {
				nameSolver.register(joinClass);
			}	
		}
		query.setNameSolver(nameSolver);
		return query;
	}
	
	@Override
	public final <T> IOrmQuery<T> findQuery(Class<T> clazz, String alias) throws OrmException {
		OrmQuery<T> query = new OrmQuery<T>(ormClassToolMap , this, clazz);
		OrmClassToolMapNameSolver nameSolver = new OrmClassToolMapNameSolver(ormClassToolMap);
		nameSolver.register(clazz, alias);
		query.setNameSolver(nameSolver);
		return query;
	}
	
	@Override
	public final IOrmCustomQuery findQuery(String selectClause, Class<?> clazz, Class<?>... joinClasses ) throws OrmException {
		OrmCustomQuery query = new OrmCustomQuery(selectClause, ormClassToolMap , this, clazz, joinClasses);
		OrmClassToolMapNameSolver nameSolver = new OrmClassToolMapNameSolver(ormClassToolMap);
		nameSolver.register(clazz);
		if (joinClasses!=null) {
			for (Class<?> joinClass : joinClasses) {
				nameSolver.register(joinClass);
			}	
		}
		query.setNameSolver(nameSolver);
		return query;
	}
	
	@Override
	public final IOrmCustomQuery findQuery(String selectClause, Class<?> clazz, String alias ) throws OrmException {
		OrmCustomQuery query = new OrmCustomQuery(selectClause, ormClassToolMap , this, clazz);
		OrmClassToolMapNameSolver nameSolver = new OrmClassToolMapNameSolver(ormClassToolMap);
		nameSolver.register(clazz, alias);
		query.setNameSolver(nameSolver);
		return query;
	}
	
	protected final IOrmClassToolMap getOrmClassToolMap() {
		return ormClassToolMap;
	}
	
	@Override
	public final <T> T find(Class<T> clazz, Object value) throws OrmException {
		return find(clazz, new Object[]{value});
	}
	
	@Override
	public final <T> T find(Class<T> clazz, Object[] values) throws OrmException {
		final IOrmClassTool<T> ormClassTool = getOrmClassToolMap().getOrmClassTool(clazz);
		IResultSetReader<T> resultSetReader = new IResultSetReader<T>() {
			@Override
			public T read(ResultSet resultSet) throws SQLException {
				if ( resultSet.next() ) {
					return ormClassTool.getOrmPersistor().mapRow("", resultSet, 0) ;
				}
				return null;
			}
		};
		ISqlPerformer sqlExec = sqlPerformer();
		sqlExec.setMaxRows(1);
		return sqlExec.query(ormClassTool.getOrmCRUDQuery().getLoadQuery(), resultSetReader, values);
	}
	
	@Override
	public final <T> void save(final T object) throws OrmException {
		
		@SuppressWarnings("unchecked")
		final IOrmClassTool<T> ormClassTool = (IOrmClassTool<T>) getOrmClassToolMap().getOrmClassTool(object.getClass());
		ISqlPerformer sqlExec = sqlPerformer();
		String sql = ormClassTool.getOrmCRUDQuery().getSaveQuery();
		Object[] args = ormClassTool.getOrmPersistor().allNotGeneratedValues(object);
		if (!ormClassTool.getOrmCRUDQuery().generatedKey()) {
			sqlExec.update(sql, args);
		} else {
			IGeneratedKeyReader generatedKeyExtractor = new IGeneratedKeyReader() {
				@Override
				public void read(ResultSet generatedKeyResultSet) throws SQLException {
					if (generatedKeyResultSet.next()) {
						ormClassTool.getOrmPersistor().updatePrimaryKey(generatedKeyResultSet, object);
					}
				}
				
				@Override
				public String[] generatedColumnNames() {
					return ormClassTool.getClassMapper().getAllGeneratedColumnDBNames();
				}
			};
			sqlExec.update(sql, generatedKeyExtractor, args);
		}
	}
	
	@Override
	public final <T> void save(final List<T> objects) throws OrmException {
		if (objects.size()==0) {
			return;
		}
		@SuppressWarnings("unchecked")
		final IOrmClassTool<T> ormClassTool = (IOrmClassTool<T>) getOrmClassToolMap().getOrmClassTool(objects.get(0).getClass());
		
		if (!ormClassTool.getOrmCRUDQuery().generatedKey()) {
			ISqlPerformer sqlExec = sqlPerformer();
			
			IPreparedStatementCreator psc = new IPreparedStatementCreator() {
				@Override
				public void set(PreparedStatement ps, int i) throws SQLException {
					T object = objects.get(i);
					int count = 0;
					for (Object value : ormClassTool.getOrmPersistor().allValues(object)) {
						ps.setObject(++count, value);
					}
				}
				@Override
				public int getBatchSize() {
					return objects.size();
				}
			};
			sqlExec.batchUpdate(ormClassTool.getOrmCRUDQuery().getSaveQuery(), psc );
		} else {
			for (T object : objects) {
				save(object);
			}
		}
	}
	
	@Override
	public final <T> void update(T object) throws OrmException {
		@SuppressWarnings("unchecked")
		IOrmClassTool<T> ormClassTool = (IOrmClassTool<T>) getOrmClassToolMap().getOrmClassTool(object.getClass());
		
		ISqlPerformer sqlExec = sqlPerformer();
		Object[] npkArgs = ormClassTool.getOrmPersistor().notPrimaryKeyValues(object);
		Object[] pkArgs = ormClassTool.getOrmPersistor().primaryKeyValues(object);
		Object[] args = new Object[ npkArgs.length + pkArgs.length ];
		int i=0;
		for (Object value : npkArgs) {
			args[i++] = value;
		}
		for (Object value : pkArgs) {
			args[i++] = value;
		}
		sqlExec.update(ormClassTool.getOrmCRUDQuery().getUpdateQuery(), args);
	}
	
	
	@Override
	public final <T> void update(final List<T> objects) throws OrmException {
		if (objects.size()==0) {
			return;
		}
		
		@SuppressWarnings("unchecked")
		final IOrmClassTool<T> ormClassTool = (IOrmClassTool<T>) getOrmClassToolMap().getOrmClassTool(objects.get(0).getClass());
		ISqlPerformer sqlExec = sqlPerformer();
		
		IPreparedStatementCreator psc = new IPreparedStatementCreator() {
			@Override
			public void set(PreparedStatement ps, int i) throws SQLException {
				T object = objects.get(i);
				int count = 0;
				for (Object value : ormClassTool.getOrmPersistor().notPrimaryKeyValues(object)) {
					ps.setObject(++count, value);
				}
				for (Object value : ormClassTool.getOrmPersistor().primaryKeyValues(object)) {
					ps.setObject(++count, value);
				}
			}
			@Override
			public int getBatchSize() {
				return objects.size();
			}
		};
		sqlExec.batchUpdate(ormClassTool.getOrmCRUDQuery().getUpdateQuery(), psc );
	}
	
	@Override
	public final IUpdate update(Class<?> clazz) throws OrmException {
		OrmUpdate update = new OrmUpdate(clazz, ormClassToolMap, this);
		OrmClassToolMapNameSolver nameSolver = new OrmClassToolMapNameSolver(ormClassToolMap);
		nameSolver.alwaysResolveWithoutAlias(true);
		nameSolver.register(clazz);
		update.setNameSolver(nameSolver);
		return update;
	}
	
	@Override
	public final IUpdate update(Class<?> clazz, String alias) throws OrmException {
		OrmUpdate update = new OrmUpdate(clazz, ormClassToolMap, this);
		OrmClassToolMapNameSolver nameSolver = new OrmClassToolMapNameSolver(ormClassToolMap);
		nameSolver.alwaysResolveWithoutAlias(true);
		nameSolver.register(clazz, alias);
		update.setNameSolver(nameSolver);
		return update;
	}
	
	@Override
	public final <T> void delete(T object) throws OrmException {
		@SuppressWarnings("unchecked")
		IOrmClassTool<T> ormClassTool = (IOrmClassTool<T>) getOrmClassToolMap().getOrmClassTool(object.getClass());
		ISqlPerformer sqlExec = sqlPerformer();
		sqlExec.update(ormClassTool.getOrmCRUDQuery().getDeleteQuery(), ormClassTool.getOrmPersistor().primaryKeyValues(object));
	}
	
	@Override
	public final <T> void delete(final List<T> objects) throws OrmException {
		if (objects.size()==0) {
			return;
		}
		@SuppressWarnings("unchecked")
		final IOrmClassTool<T> ormClassTool = (IOrmClassTool<T>) getOrmClassToolMap().getOrmClassTool(objects.get(0).getClass());
		ISqlPerformer sqlExec = sqlPerformer();
		
		IPreparedStatementCreator psc = new IPreparedStatementCreator() {
			@Override
			public void set(PreparedStatement ps, int i) throws SQLException {
				T object = objects.get(i);
				int count = 0;
				for (Object value : ormClassTool.getOrmPersistor().primaryKeyValues(object)) {
					ps.setObject(++count, value);
				}
			}
			@Override
			public int getBatchSize() {
				return objects.size();
			}
		};
		sqlExec.batchUpdate(ormClassTool.getOrmCRUDQuery().getDeleteQuery(), psc );
	}
	
	@Override
	public final IDelete delete(Class<?> clazz) throws OrmException {
		OrmDelete delete = new OrmDelete(clazz, ormClassToolMap, this);
		OrmClassToolMapNameSolver nameSolver = new OrmClassToolMapNameSolver(ormClassToolMap);
		nameSolver.register(clazz);
		nameSolver.alwaysResolveWithoutAlias(true);
		delete.setNameSolver(nameSolver);
		return delete;
	}
	
	@Override
	public final IDelete delete(Class<?> clazz, String alias) throws OrmException {
		OrmDelete delete = new OrmDelete(clazz, ormClassToolMap, this);
		OrmClassToolMapNameSolver nameSolver = new OrmClassToolMapNameSolver(ormClassToolMap);
		nameSolver.register(clazz, alias);
		nameSolver.alwaysResolveWithoutAlias(true);
		delete.setNameSolver(nameSolver);
		return delete;
	}

	@Override
	public final IScriptExecutor scriptExecutor() throws OrmException {
		return new ScriptExecutor(this);
	}
	
	@Override
	public final ISqlExecutor sqlExecutor() {
		return new SqlExecutor(this);
	}
	
}