package com.jpattern.orm.session;

import java.util.List;

import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.query.IDelete;
import com.jpattern.orm.query.IOrmCustomQuery;
import com.jpattern.orm.query.IOrmQuery;
import com.jpattern.orm.query.ISqlExecutor;
import com.jpattern.orm.query.IUpdate;
import com.jpattern.orm.script.IScriptExecutor;

/**
 * 
 * @author Francesco Cina
 *
 * 21/mag/2011
 * 
 * A connection wrapper object witch the detail of the real used connection object 
 */
public interface ISession {
	
	/**
	 * Open a new Transaction
	 * @return
	 * @throws OrmException
	 */
	ITransaction beginTransaction() throws OrmException;

	/**
	 * Load a registered Object from the database
	 * @param <T>
	 * @param clazz The Class of the object to load
	 * @param idValue the value of the identifying column of the object
	 * @return
	 * @throws OrmException
	 */
	<T> T find(Class<T> clazz, Object idValue) throws OrmException;
	
	/**
	 * Load a registered Object from the database
	 * @param <T>
	 * @param clazz The Class of the object to load
	 * @param idValues an ordered array with the values of the identifying columns of the object
	 * @return
	 * @throws OrmException
	 */
	<T> T find(Class<T> clazz, Object[] idValues) throws OrmException;

	/**
	 * Persist the new object in the database
	 * @param <T>
	 * @param object
	 * @throws OrmException
	 */
	<T> void save(T object) throws OrmException;

	/**
	 * Persist the new objects in the database
	 * @param <T>
	 * @param objects the objects to persist
	 * @throws OrmException
	 */
	<T> void save(List<T> objects) throws OrmException;
	
	/**
	 * Update the values of an existing object in the database
	 * @param <T>
	 * @param object
	 * @throws OrmException
	 */
	<T> void update(T object) throws OrmException;

	/**
	 * Update the values of the existing objects in the database
	 * @param <T>
	 * @param objects the objects to update
	 * @throws OrmException
	 */
	<T> void update(List<T> objects) throws OrmException;
	
	/**
	 * Update the objects of a specific TABLE 
	 * @param clazz the TABLE related Class
	 * @throws OrmException
	 */
	IUpdate update(Class<?> clazz) throws OrmException;
	
	/**
	 * Update the objects of a specific TABLE
	 * @param clazz the TABLE related Class
	 * @param alias The alias of the class in the query.
	 * @throws OrmException
	 */
	IUpdate update(Class<?> clazz, String alias) throws OrmException;
	
	
	/**
	 * Delete one object from the database
	 * @param <T>
	 * @param object
	 * @throws OrmException
	 */
	<T> void delete(T object) throws OrmException;
	
	/**
	 * Delete the objects from the database
	 * @param <T>
	 * @param objects the objects to delete
	 * @throws OrmException
	 */
	<T> void delete(List<T> object) throws OrmException;
	
	/**
	 * Delete the objects of a specific table
	 * @param clazz the TABLE related Class
	 * @throws OrmException
	 */
	IDelete delete(Class<?> clazz) throws OrmException;
	
	/**
	 * Delete the objects of a specific table
	 * @param clazz the TABLE related Class
	 * @param alias The alias of the class in the query.
	 * @throws OrmException
	 */
	IDelete delete(Class<?> clazz, String alias) throws OrmException;
	
	/**
	 * Create a new query
	 * @param <T>
	 * @param clazz The class of the object that will be retrieved by the query execution. The simple class name will be used as alias for the class
	 * @param joinClasses a list of class that will be added in the 'from' using comma separation. For every class the simple class name will be used as alias.
	 * @return
	 * @throws OrmException
	 */
	<T> IOrmQuery<T> findQuery(Class<T> clazz, Class<?>... joinClasses ) throws OrmException;
	
	/**
	 * Create a new query
	 * @param <T>
	 * @param clazz The class of the object that will be retrieved by the query execution.
	 * @param alias The alias of the class in the query.
	 * @return
	 * @throws OrmException
	 */
	<T> IOrmQuery<T> findQuery(Class<T> clazz, String alias ) throws OrmException;
	
	/**
	 * Create a new custom query
	 * @param selectClause the select clause of the query (do not use the "select" keyword)
	 * @param clazz The class of the object that will be retrieved by the query execution. The simple class name will be used as alias for the class
	 * @param joinClasses a list of class that will be added in the 'from' using comma separation. For every class the simple class name will be used as alias.
	 * @return
	 * @throws OrmException
	 */
	IOrmCustomQuery findQuery(String selectClause, Class<?> clazz, Class<?>... joinClasses ) throws OrmException;
	
	/**
	 * Create a new query
	 * @param selectClause the select clause of the query (do not use the "select" keyword)
	 * @param clazz The class of the object that will be retrieved by the query execution.
	 * @param alias The alias of the class in the query.
	 * @return
	 * @throws OrmException
	 */
	IOrmCustomQuery findQuery(String selectClause, Class<?> clazz, String alias ) throws OrmException;
	
	/**
	 * Close the current session and release the connection to the database.
	 */
	void close() throws OrmException;
	
	/**
	 * An executor to perform any kind of plain SQL statements.
	 * @return
	 */
	ISqlExecutor sqlExecutor();
	
	/**
	 * A script executor useful to execute multiple sql statement from files.
	 * @return
	 * @throws OrmException
	 */
	IScriptExecutor scriptExecutor() throws OrmException;
}
