package com.jpattern.orm.session;

import java.math.BigDecimal;
import java.util.List;

import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.exception.OrmNotUniqueResultException;

/**
 * 
 * @author Francesco Cina
 *
 * 02/lug/2011
 * 
 * An executor to perform plain SQL queries 
 */
public interface ISqlPerformer {

	/**
	 * Set the maximum number of rows returnd by the execution of the sql query
	 */
	public void setMaxRows(int maxRows);

	/**
	 * Return the maximum number of rows specified for this sql query.
	 */
	public int getMaxRows();
	
	/**
	 * Set the query timeout for the statements.
	 */
	public void setQueryTimeout(int queryTimeout);

	/**
	 * Return the query timeout for the statements.
	 */
	public int getQueryTimeout();
	
	/**
	 * Issue a single SQL execute, typically a DDL statement.
	 * @param sql static SQL to execute
	 */
	void execute(String sql) throws OrmException;
	
	/**
	 * Execute a query given static SQL, reading the ResultSet with a
	 * IResultSetReader.
	 * @param sql SQL query to execute
	 * @param rse object that will extract all rows of results
	 * @param args arguments to bind to the query
	 * @return an arbitrary result object, as returned by the IResultSetExtractor
	 */
	<T> T query(String sql, IResultSetReader<T> rse, Object... args) throws OrmException;
	
	/**
	 * Execute a query given static SQL and read the result as an int value (return null if no rows are returned)
	 * @param sql SQL query to execute
	 * @param args arguments to bind to the query
	 * @return 
	 * @throws OrmNotUniqueResultException if more than one row is returned from the query
	 */
	Integer queryForInt(String sql, Object... args) throws OrmException, OrmNotUniqueResultException;
	
	/**
	 * Execute a query given static SQL and read the result as an long value (return null if no rows are returned)
	 * @param sql SQL query to execute
	 * @param args arguments to bind to the query
	 * @return 
	 * @throws OrmNotUniqueResultException if more than one row is returned from the query
	 */
	Long queryForLong(String sql, Object... args) throws OrmException, OrmNotUniqueResultException;

	/**
	 * Execute a query given static SQL and read the result as a double value (return null if no rows are returned)
	 * @param sql SQL query to execute
	 * @param args arguments to bind to the query
	 * @return 
	 * @throws OrmNotUniqueResultException if more than one row is returned from the query
	 */
	Double queryForDouble(String sql, Object... args) throws OrmException, OrmNotUniqueResultException;
	
	/**
	 * Execute a query given static SQL and read the result as a float value (return null if no rows are returned)
	 * @param sql SQL query to execute
	 * @param args arguments to bind to the query
	 * @return 
	 * @throws OrmNotUniqueResultException if more than one row is returned from the query
	 */
	Float queryForFloat(String sql, Object... args) throws OrmException, OrmNotUniqueResultException;
	
	/**
	 * Execute a query given static SQL and read the result as a String value (return null if no rows are returned)
	 * @param sql SQL query to execute
	 * @param args arguments to bind to the query
	 * @return 
	 * @throws OrmNotUniqueResultException if more than one row is returned from the query
	 */
	String queryForString(String sql, Object... args) throws OrmException, OrmNotUniqueResultException;

	/**
	 * Execute a query given static SQL and read the result as a boolean value (return null if no rows are returned)
	 * @param sql SQL query to execute
	 * @param args arguments to bind to the query
	 * @return 
	 * @throws OrmNotUniqueResultException if more than one row is returned from the query
	 */
	Boolean queryForBoolean(String sql, Object... args) throws OrmException, OrmNotUniqueResultException;

	/**
	 * Execute a query given static SQL and read the result as a BigDecimal value (return null if no rows are returned)
	 * @param sql SQL query to execute
	 * @param args arguments to bind to the query
	 * @return 
	 * @throws OrmNotUniqueResultException if more than one row is returned from the query
	 */
	BigDecimal queryForBigDecimal(String sql, Object... args) throws OrmException, OrmNotUniqueResultException;

	/**
	 * Execute a query given static SQL and read the result creating an ordered array with the extracted column values (return null if no rows are returned)
	 * @param sql SQL query to execute
	 * @param args arguments to bind to the query
	 * @return 
	 * @throws OrmNotUniqueResultException if more than one row is returned from the query
	 */
	Object[] queryForArray(String sql, Object... args) throws OrmException, OrmNotUniqueResultException;
	
	/**
	 * Execute a query given static SQL and read the result creating a List of all the ordered arrays with the extracted column values for every row.
	 * @param sql SQL query to execute
	 * @param args arguments to bind to the query
	 * @return 
	 */
	List<Object[]> queryForList(String sql, Object... args) throws OrmException;
	
	/**
	 * Perform a single SQL update operation (such as an insert, update or delete statement).
	 * @param sql static SQL to execute
	 * @param args arguments to bind to the query
	 * @return the number of rows affected
	 */
	int update(String sql, Object... args) throws OrmException;
	
	/**
	 * Issue an update statement using a PreparedStatementCreator to provide SQL and
	 * any required parameters. Generated keys can be read using the IGeneratedKeyReader.
	 * @param psc object that provides SQL and any necessary parameters
	 * @param generatedKeyReader IGeneratedKeyReader to read the generated key
	 * @return the number of rows affected
	 */
	int update(String sql, IGeneratedKeyReader generatedKeyReader, Object... args) throws OrmException;

	/**
	 * Issue multiple SQL updates on a single JDBC Statement using batching.
	 * @param sql defining a List of SQL statements that will be executed.
	 * @return an array of the number of rows affected by each statement
	 */
	int[] batchUpdate(List<String> sqls) throws OrmException;
	
	
	
	/**
	 * Issue multiple SQL updates on a single JDBC Statement using batching.
	 * The same query is executed for every Object array present in the args list
	 * which is the list of arguments to bind to the query.
	 * @param sql defining a List of SQL statements that will be executed.
	 * @param args defining a List of Object arrays to bind to the query. 
	 * @return an array of the number of rows affected by each statement
	 */
	int[] batchUpdate(String sql, List<Object[]> args) throws OrmException;
	
	/**
	 * Issue multiple SQL updates on a single JDBC Statement using batching.
	 * The values on the generated PreparedStatement are set using an IPreparedStatementCreator.
	 * @param sql defining a List of SQL statements that will be executed.
	 * @param psc the creator to bind values on the PreparedStatement 
	 * @return an array of the number of rows affected by each statement
	 */
	int[] batchUpdate(String sql, IPreparedStatementCreator psc) throws OrmException;
	
}
