/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.java.debug.core.protocol;

import com.microsoft.java.debug.core.adapter.AdapterUtils;
import com.microsoft.java.debug.core.adapter.ErrorCode;
import com.microsoft.java.debug.core.protocol.Events;
import com.microsoft.java.debug.core.protocol.IProtocolServer;
import com.microsoft.java.debug.core.protocol.JsonUtils;
import com.microsoft.java.debug.core.protocol.Messages;
import io.reactivex.disposables.Disposable;
import io.reactivex.schedulers.Schedulers;
import io.reactivex.subjects.PublishSubject;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class AbstractProtocolServer
implements IProtocolServer {
    private static final Logger logger = Logger.getLogger("java-debug");
    private static final int BUFFER_SIZE = 4096;
    private static final String TWO_CRLF = "\r\n\r\n";
    private static final Pattern CONTENT_LENGTH_MATCHER = Pattern.compile("Content-Length: (\\d+)");
    private static final Charset PROTOCOL_ENCODING = StandardCharsets.UTF_8;
    protected boolean terminateSession = false;
    private Reader reader;
    private Writer writer;
    private ByteBuffer rawData;
    private int contentLength = -1;
    private AtomicInteger sequenceNumber = new AtomicInteger(1);
    private boolean isValidDAPRequest = true;
    private PublishSubject<Messages.Response> responseSubject = PublishSubject.create();
    private PublishSubject<Messages.Request> requestSubject = PublishSubject.create();

    public AbstractProtocolServer(InputStream input, OutputStream output) {
        this.reader = new BufferedReader(new InputStreamReader(input, PROTOCOL_ENCODING));
        this.writer = new PrintWriter(new BufferedWriter(new OutputStreamWriter(output, PROTOCOL_ENCODING)));
        this.contentLength = -1;
        this.rawData = new ByteBuffer();
        this.requestSubject.observeOn(Schedulers.newThread()).subscribe(request -> {
            try {
                this.dispatchRequest((Messages.Request)request);
            }
            catch (Exception e) {
                logger.log(Level.SEVERE, String.format("Dispatch debug protocol error: %s", e.toString()), e);
            }
        });
    }

    public void run() {
        char[] buffer = new char[4096];
        try {
            int read;
            while (!this.terminateSession && (read = this.reader.read(buffer, 0, 4096)) != -1) {
                this.rawData.append(new String(buffer, 0, read).getBytes(PROTOCOL_ENCODING));
                this.processData();
            }
        }
        catch (IOException e) {
            logger.log(Level.SEVERE, String.format("Read data from io exception: %s", e.toString()), e);
        }
        this.requestSubject.onComplete();
    }

    public void stop() {
        this.terminateSession = true;
    }

    private void sendMessage(Messages.ProtocolMessage message) {
        message.seq = this.sequenceNumber.getAndIncrement();
        String jsonMessage = JsonUtils.toJson(message);
        byte[] jsonBytes = jsonMessage.getBytes(PROTOCOL_ENCODING);
        String header = String.format("Content-Length: %d%s", jsonBytes.length, TWO_CRLF);
        byte[] headerBytes = header.getBytes(PROTOCOL_ENCODING);
        ByteBuffer data = new ByteBuffer();
        data.append(headerBytes);
        data.append(jsonBytes);
        String utf8Data = data.getString(PROTOCOL_ENCODING);
        try {
            if (message instanceof Messages.Request) {
                logger.fine("\n[[REQUEST]]\n" + utf8Data);
            } else if (message instanceof Messages.Event) {
                logger.fine("\n[[EVENT]]\n" + utf8Data);
            } else {
                logger.fine("\n[[RESPONSE]]\n" + utf8Data);
            }
            this.writer.write(utf8Data);
            this.writer.flush();
        }
        catch (IOException e) {
            logger.log(Level.SEVERE, String.format("Write data to io exception: %s", e.toString()), e);
        }
    }

    @Override
    public void sendEvent(Events.DebugEvent event) {
        this.sendMessage(new Messages.Event(event.type, event));
    }

    @Override
    public void sendResponse(Messages.Response response) {
        this.sendMessage(response);
    }

    @Override
    public CompletableFuture<Messages.Response> sendRequest(Messages.Request request) {
        return this.sendRequest(request, 0L);
    }

    @Override
    public CompletableFuture<Messages.Response> sendRequest(Messages.Request request, long timeout) {
        Disposable[] disposable;
        final CompletableFuture<Messages.Response> future = new CompletableFuture<Messages.Response>();
        Timer timer = new Timer();
        disposable = new Disposable[]{this.responseSubject.filter(response -> response.request_seq == request.seq).take(1L).observeOn(Schedulers.newThread()).subscribe(response -> {
            try {
                timer.cancel();
                future.complete((Messages.Response)response);
                if (disposable[0] != null) {
                    disposable[0].dispose();
                }
            }
            catch (Exception e) {
                logger.log(Level.SEVERE, String.format("Handle response error: %s", e.toString()), e);
            }
        })};
        this.sendMessage(request);
        if (timeout > 0L) {
            try {
                timer.schedule(new TimerTask(){

                    @Override
                    public void run() {
                        if (disposable[0] != null) {
                            disposable[0].dispose();
                        }
                        future.completeExceptionally(new TimeoutException("timeout"));
                    }
                }, timeout);
            }
            catch (IllegalStateException illegalStateException) {
                // empty catch block
            }
        }
        return future;
    }

    private void processData() {
        while (true) {
            Matcher matcher;
            if (this.contentLength >= 0 && this.rawData.length() >= this.contentLength) {
                byte[] buf = this.rawData.removeFirst(this.contentLength);
                this.contentLength = -1;
                String messageData = new String(buf, PROTOCOL_ENCODING);
                try {
                    Messages.ProtocolMessage message = JsonUtils.fromJson(messageData, Messages.ProtocolMessage.class);
                    logger.fine(String.format("\n[%s]\n%s", message.type, messageData));
                    if (message.type.equals("request")) {
                        Messages.Request request = JsonUtils.fromJson(messageData, Messages.Request.class);
                        if (this.isValidDAPRequest) {
                            this.requestSubject.onNext((Object)request);
                            continue;
                        }
                        Messages.Response response = new Messages.Response(request.seq, request.command);
                        this.sendResponse(AdapterUtils.setErrorResponse(response, ErrorCode.INVALID_DAP_HEADER, String.format("'%s' request is rejected due to not being a valid DAP message.", request.command)));
                        continue;
                    }
                    if (!message.type.equals("response")) continue;
                    Messages.Response response = JsonUtils.fromJson(messageData, Messages.Response.class);
                    this.responseSubject.onNext((Object)response);
                }
                catch (Exception ex) {
                    logger.log(Level.SEVERE, String.format("Error parsing message: %s", ex.toString()), ex);
                }
                continue;
            }
            String rawMessage = this.rawData.getString(PROTOCOL_ENCODING);
            int idx = rawMessage.indexOf(TWO_CRLF);
            if (idx == -1 || !(matcher = CONTENT_LENGTH_MATCHER.matcher(rawMessage)).find()) break;
            String contentLengthText = matcher.group(1);
            this.contentLength = Integer.parseInt(contentLengthText);
            String headerMessage = rawMessage.substring(0, idx + TWO_CRLF.length());
            int headerByteLength = headerMessage.getBytes(PROTOCOL_ENCODING).length;
            this.rawData.removeFirst(headerByteLength);
            int expectedHeaderLength = 16 + contentLengthText.length();
            int actualHeaderLength = idx;
            if (expectedHeaderLength != actualHeaderLength) {
                this.isValidDAPRequest = false;
                logger.log(Level.SEVERE, String.format("Illegal DAP request is detected: %s", headerMessage));
                continue;
            }
            this.isValidDAPRequest = true;
        }
    }

    protected abstract void dispatchRequest(Messages.Request var1);

    class ByteBuffer {
        private byte[] buffer = new byte[0];

        public int length() {
            return this.buffer.length;
        }

        public String getString(Charset cs) {
            return new String(this.buffer, cs);
        }

        public void append(byte[] b) {
            this.append(b, b.length);
        }

        public void append(byte[] b, int length) {
            byte[] newBuffer = new byte[this.buffer.length + length];
            System.arraycopy(this.buffer, 0, newBuffer, 0, this.buffer.length);
            System.arraycopy(b, 0, newBuffer, this.buffer.length, length);
            this.buffer = newBuffer;
        }

        public byte[] removeFirst(int n) {
            byte[] b = new byte[n];
            System.arraycopy(this.buffer, 0, b, 0, n);
            byte[] newBuffer = new byte[this.buffer.length - n];
            System.arraycopy(this.buffer, n, newBuffer, 0, this.buffer.length - n);
            this.buffer = newBuffer;
            return b;
        }
    }
}

