/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * <p>
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * <p>
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * <p>
 * 3. Neither the name of Liferay, Inc. nor the names of its contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package com.wedeploy.api.sdk;

/**
 * Common content types.
 */
public class ContentType {

	public static final ContentType TEXT = new ContentType(
		"text/plain", "UTF-8");
	public static final ContentType HTML = new ContentType(
		"text/html", "UTF-8");
	public static final ContentType JSON = new ContentType(
		"application/json", "UTF-8");
	public static final ContentType FORM_URLENCODED = new ContentType(
		"application/x-www-form-urlencoded", "UTF-8");
	public static final ContentType MULTIPART_FORM = new ContentType(
		"multipart/form-data", "UTF-8");

	public ContentType(String contentType) {
		ContentTypeHeaderResolver cthr = new ContentTypeHeaderResolver(
			contentType);

		this.contentType = cthr.getMimeType();
		this.charset = cthr.getEncoding();

		if (charset == null) {
			completeContentType = this.contentType;
		} else {
			completeContentType = this.contentType + "; charset=" + charset;
		}
	}

	public ContentType(String contentType, String charset) {
		this.contentType = contentType;
		this.charset = charset;

		if (charset == null) {
			completeContentType = contentType;
		} else {
			completeContentType = contentType + "; charset=" + charset;
		}
	}

	/**
	 * Returns content-type value.
	 */
	public String contentType() {
		return contentType;
	}

	/**
	 * Returns content-type charset.
	 */
	public String charset() {
		return charset;
	}

	@Override
	public String toString() {
		return completeContentType;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		ContentType that = (ContentType) o;

		return completeContentType.equals(that.completeContentType);
	}

	/**
	 * Returns <code>true</code> if content-type is equals, ignoring the
	 * charset.
	 */
	public boolean isSame(String type) {
		if (type == null) {
			return false;
		}

		int ndx = type.indexOf(';');

		if (ndx != -1) {
			type = type.substring(0, ndx);
		}

		return type.equals(this.contentType);
	}

	@Override
	public int hashCode() {
		return completeContentType.hashCode();
	}

	private final String completeContentType;
	private final String contentType;
	private final String charset;

}
