/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * <p>
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * <p>
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * <p>
 * 3. Neither the name of Liferay, Inc. nor the names of its contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package com.wedeploy.api.serializer;

import com.wedeploy.api.sdk.ContentType;

import java.util.List;
import java.util.Map;

/**
 * WeDeploy Parser deserializes the input string into an object. There are two
 * types of parsing:
 * <ul>
 * <li>to common JDK types, like <code>List</code>, <code>Map</code>,
 * <code>String</code>...</li>
 * <li>to specific types.</li>
 * </ul>
 */
public interface Parser {

	public static Parser get() {
		return SerializerEngine.instance().parser();
	}

	public static Parser get(ContentType contentType) {
		return get(contentType.contentType());
	}

	public static Parser get(String contentType) {
		return SerializerEngine.instance().parser(contentType);
	}

	/**
	 * Parses string to Java common type.
	 */
	public <T> T parse(String string);

	/**
	 * Parses string to given type.
	 */
	public <T> T parse(String string, Class<T> type);

	/**
	 * Parses string to a list of given type.
	 */
	public <T> List<T> parseAsList(String string, Class<T> componentType);

	/**
	 * Parses string to a map of given key and value types.
	 */
	public <K, V> Map<K, V> parseAsMap(
		String string, Class<K> keyType, Class<V> valueType);

	/**
	 * {@link #parse(String) Parses} string without throwing an exception and
	 * returning <code>null</code> on failures.
	 */
	public default <T> T parseSilently(String string) {
		try {
			return parse(string);
		} catch (Exception ignore) {
			return null;
		}
	}

	public default <T> T parseSilently(String string, Class<T> type) {
		try {
			return parse(string, type);
		} catch (Exception ignore) {
			return null;
		}
	}

}
