/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * <p>
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * <p>
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * <p>
 * 3. Neither the name of Liferay, Inc. nor the names of its contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package com.wedeploy.api.serializer;

import java.util.HashMap;
import java.util.Map;

/**
 * Main class that performs bindings and holds instances of serializer and parser.
 */
public class SerializerEngine {

	/**
	 * Returns singleton instance of the engine.
	 */
	public static SerializerEngine instance() {
		return instance;
	}

	public Parser parser() {
		return defaultEngines().getParser();
	}

	/**
	 * Returns te {@link Parser} instance.
	 */
	public Parser parser(String contentType) {
		Engines engines = lookupEngines(contentType);

		return engines.getParser();
	}

	/**
	 * Manual registration of engines.
	 */
	public void registerEngines(
		String serializerType, Engines engines, boolean isDefault) {

		enginesMap.put(serializerType, engines);

		if (isDefault) {
			defaultEngines = engines;
		}
	}

	public Serializer serializer() {
		return defaultEngines().getSerializer();
	}

	/**
	 * Returns the {@link Serializer} instance.
	 */
	public Serializer serializer(String contentType) {
		Engines engines = lookupEngines(contentType);

		return engines.getSerializer();
	}

	protected SerializerEngine() {
	}

	protected void clearEngines() {
		defaultEngines = null;
		enginesMap.clear();
	}

	protected Engines defaultEngines() {
		if (defaultEngines == null) {
			throw new SerializerException("Default content type is not set.");
		}

		return defaultEngines;
	}

	protected Engines lookupEngines(String contentType) {
		Engines engines = enginesMap.get(contentType);

		if (engines == null) {
			if (defaultEngines == null) {
				throw new SerializerException(
					"Engine for content type not found: " + contentType);
			}

			return defaultEngines;
		}

		return engines;
	}

	private static final SerializerEngine instance = new SerializerEngine();

	private Engines defaultEngines;
	private Map<String, Engines> enginesMap = new HashMap<>();

}
