/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.remoteconfig.internal;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageManager;
import android.util.Log;
import androidx.annotation.GuardedBy;
import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;
import com.google.android.gms.common.util.AndroidUtilsLight;
import com.google.android.gms.common.util.Clock;
import com.google.android.gms.common.util.DefaultClock;
import com.google.android.gms.common.util.Hex;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.Tasks;
import com.google.firebase.FirebaseApp;
import com.google.firebase.installations.FirebaseInstallationsApi;
import com.google.firebase.installations.InstallationTokenResult;
import com.google.firebase.remoteconfig.ConfigUpdate;
import com.google.firebase.remoteconfig.ConfigUpdateListener;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigClientException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigServerException;
import com.google.firebase.remoteconfig.internal.ConfigAutoFetch;
import com.google.firebase.remoteconfig.internal.ConfigCacheClient;
import com.google.firebase.remoteconfig.internal.ConfigFetchHandler;
import com.google.firebase.remoteconfig.internal.ConfigMetadataClient;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Date;
import java.util.HashMap;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.json.JSONObject;

public class ConfigRealtimeHttpClient {
    @VisibleForTesting
    static final int[] BACKOFF_TIME_DURATIONS_IN_MINUTES = new int[]{2, 4, 8, 16, 32, 64, 128, 256};
    private static final Pattern GMP_APP_ID_PATTERN = Pattern.compile("^[^:]+:([0-9]+):(android|ios|web):([0-9a-f]+)");
    private static final String API_KEY_HEADER = "X-Goog-Api-Key";
    private static final String X_ANDROID_PACKAGE_HEADER = "X-Android-Package";
    private static final String X_ANDROID_CERT_HEADER = "X-Android-Cert";
    private static final String X_GOOGLE_GFE_CAN_RETRY = "X-Google-GFE-Can-Retry";
    private static final String INSTALLATIONS_AUTH_TOKEN_HEADER = "X-Goog-Firebase-Installations-Auth";
    private static final String X_ACCEPT_RESPONSE_STREAMING = "X-Accept-Response-Streaming";
    @GuardedBy(value="this")
    private final Set<ConfigUpdateListener> listeners;
    @GuardedBy(value="this")
    private boolean isHttpConnectionRunning;
    @GuardedBy(value="this")
    private int httpRetriesRemaining;
    @GuardedBy(value="this")
    private boolean isRealtimeDisabled;
    private boolean isInBackground;
    private final int ORIGINAL_RETRIES = 8;
    private final ScheduledExecutorService scheduledExecutorService;
    private final ConfigFetchHandler configFetchHandler;
    private final FirebaseApp firebaseApp;
    private final FirebaseInstallationsApi firebaseInstallations;
    ConfigCacheClient activatedCache;
    private final Context context;
    private final String namespace;
    private final Random random;
    private final Clock clock;
    private final ConfigMetadataClient metadataClient;

    public ConfigRealtimeHttpClient(FirebaseApp firebaseApp, FirebaseInstallationsApi firebaseInstallations, ConfigFetchHandler configFetchHandler, ConfigCacheClient activatedCache, Context context, String namespace, Set<ConfigUpdateListener> listeners, ConfigMetadataClient metadataClient, ScheduledExecutorService scheduledExecutorService) {
        this.listeners = listeners;
        this.isHttpConnectionRunning = false;
        this.scheduledExecutorService = scheduledExecutorService;
        this.random = new Random();
        this.httpRetriesRemaining = Math.max(8 - metadataClient.getRealtimeBackoffMetadata().getNumFailedStreams(), 1);
        this.clock = DefaultClock.getInstance();
        this.firebaseApp = firebaseApp;
        this.configFetchHandler = configFetchHandler;
        this.firebaseInstallations = firebaseInstallations;
        this.activatedCache = activatedCache;
        this.context = context;
        this.namespace = namespace;
        this.metadataClient = metadataClient;
        this.isRealtimeDisabled = false;
        this.isInBackground = false;
    }

    private static String extractProjectNumberFromAppId(String gmpAppId) {
        Matcher matcher = GMP_APP_ID_PATTERN.matcher(gmpAppId);
        return matcher.matches() ? matcher.group(1) : null;
    }

    private String getFingerprintHashForPackage() {
        try {
            byte[] hash = AndroidUtilsLight.getPackageCertificateHashBytes(this.context, this.context.getPackageName());
            if (hash == null) {
                Log.e("FirebaseRemoteConfig", "Could not get fingerprint hash for package: " + this.context.getPackageName());
                return null;
            }
            return Hex.bytesToStringUppercase(hash, false);
        }
        catch (PackageManager.NameNotFoundException e) {
            Log.i("FirebaseRemoteConfig", "No such package: " + this.context.getPackageName());
            return null;
        }
    }

    private void setCommonRequestHeaders(HttpURLConnection httpURLConnection, String authToken) {
        httpURLConnection.setRequestProperty(INSTALLATIONS_AUTH_TOKEN_HEADER, authToken);
        httpURLConnection.setRequestProperty(API_KEY_HEADER, this.firebaseApp.getOptions().getApiKey());
        httpURLConnection.setRequestProperty(X_ANDROID_PACKAGE_HEADER, this.context.getPackageName());
        httpURLConnection.setRequestProperty(X_ANDROID_CERT_HEADER, this.getFingerprintHashForPackage());
        httpURLConnection.setRequestProperty(X_GOOGLE_GFE_CAN_RETRY, "yes");
        httpURLConnection.setRequestProperty(X_ACCEPT_RESPONSE_STREAMING, "true");
        httpURLConnection.setRequestProperty("Content-Type", "application/json");
        httpURLConnection.setRequestProperty("Accept", "application/json");
    }

    private JSONObject createRequestBody(String installationId) {
        HashMap<String, String> body = new HashMap<String, String>();
        body.put("project", ConfigRealtimeHttpClient.extractProjectNumberFromAppId(this.firebaseApp.getOptions().getApplicationId()));
        body.put("namespace", this.namespace);
        body.put("lastKnownVersionNumber", Long.toString(this.configFetchHandler.getTemplateVersionNumber()));
        body.put("appId", this.firebaseApp.getOptions().getApplicationId());
        body.put("sdkVersion", "21.6.0");
        body.put("appInstanceId", installationId);
        return new JSONObject(body);
    }

    @SuppressLint(value={"VisibleForTests"})
    public void setRequestParams(HttpURLConnection httpURLConnection, String installationId, String authtoken) throws IOException {
        httpURLConnection.setRequestMethod("POST");
        this.setCommonRequestHeaders(httpURLConnection, authtoken);
        byte[] body = this.createRequestBody(installationId).toString().getBytes("utf-8");
        BufferedOutputStream outputStream = new BufferedOutputStream(httpURLConnection.getOutputStream());
        ((OutputStream)outputStream).write(body);
        ((OutputStream)outputStream).flush();
        ((OutputStream)outputStream).close();
    }

    private synchronized void propagateErrors(FirebaseRemoteConfigException exception) {
        for (ConfigUpdateListener listener2 : this.listeners) {
            listener2.onError(exception);
        }
    }

    @SuppressLint(value={"VisibleForTests"})
    public int getNumberOfFailedStreams() {
        return this.metadataClient.getRealtimeBackoffMetadata().getNumFailedStreams();
    }

    @SuppressLint(value={"VisibleForTests"})
    public Date getBackoffEndTime() {
        return this.metadataClient.getRealtimeBackoffMetadata().getBackoffEndTime();
    }

    private void updateBackoffMetadataWithLastFailedStreamConnectionTime(Date lastFailedRealtimeStreamTime) {
        int numFailedStreams = this.metadataClient.getRealtimeBackoffMetadata().getNumFailedStreams();
        long backoffDurationInMillis = this.getRandomizedBackoffDurationInMillis(++numFailedStreams);
        Date backoffEndTime = new Date(lastFailedRealtimeStreamTime.getTime() + backoffDurationInMillis);
        this.metadataClient.setRealtimeBackoffMetadata(numFailedStreams, backoffEndTime);
    }

    private long getRandomizedBackoffDurationInMillis(int numFailedStreams) {
        int backoffIndex = BACKOFF_TIME_DURATIONS_IN_MINUTES.length;
        if (numFailedStreams < backoffIndex) {
            backoffIndex = numFailedStreams;
        }
        long timeOutDurationInMillis = TimeUnit.MINUTES.toMillis(BACKOFF_TIME_DURATIONS_IN_MINUTES[backoffIndex - 1]);
        return timeOutDurationInMillis / 2L + (long)this.random.nextInt((int)timeOutDurationInMillis);
    }

    private synchronized void enableBackoff() {
        this.isRealtimeDisabled = true;
    }

    private synchronized boolean canMakeHttpStreamConnection() {
        return !this.listeners.isEmpty() && !this.isHttpConnectionRunning && !this.isRealtimeDisabled && !this.isInBackground;
    }

    private String getRealtimeURL(String namespace) {
        return String.format("https://firebaseremoteconfigrealtime.googleapis.com/v1/projects/%s/namespaces/%s:streamFetchInvalidations", ConfigRealtimeHttpClient.extractProjectNumberFromAppId(this.firebaseApp.getOptions().getApplicationId()), namespace);
    }

    private URL getUrl() {
        URL realtimeURL = null;
        try {
            realtimeURL = new URL(this.getRealtimeURL(this.namespace));
        }
        catch (MalformedURLException ex) {
            Log.e("FirebaseRemoteConfig", "URL is malformed");
        }
        return realtimeURL;
    }

    @SuppressLint(value={"VisibleForTests"})
    public Task<HttpURLConnection> createRealtimeConnection() {
        Task<InstallationTokenResult> installationAuthTokenTask = this.firebaseInstallations.getToken(false);
        Task<String> installationIdTask = this.firebaseInstallations.getId();
        return Tasks.whenAllComplete(installationAuthTokenTask, installationIdTask).continueWithTask(this.scheduledExecutorService, unusedCompletedInstallationTasks -> {
            HttpURLConnection httpURLConnection;
            if (!installationAuthTokenTask.isSuccessful()) {
                return Tasks.forException(new FirebaseRemoteConfigClientException("Firebase Installations failed to get installation auth token for config update listener connection.", installationAuthTokenTask.getException()));
            }
            if (!installationIdTask.isSuccessful()) {
                return Tasks.forException(new FirebaseRemoteConfigClientException("Firebase Installations failed to get installation ID for config update listener connection.", installationIdTask.getException()));
            }
            try {
                URL realtimeUrl = this.getUrl();
                httpURLConnection = (HttpURLConnection)realtimeUrl.openConnection();
                String installationAuthToken = ((InstallationTokenResult)installationAuthTokenTask.getResult()).getToken();
                String installationId = (String)installationIdTask.getResult();
                this.setRequestParams(httpURLConnection, installationId, installationAuthToken);
            }
            catch (IOException ex) {
                return Tasks.forException(new FirebaseRemoteConfigClientException("Failed to open HTTP stream connection", ex));
            }
            return Tasks.forResult(httpURLConnection);
        });
    }

    public void startHttpConnection() {
        this.makeRealtimeHttpConnection(0L);
    }

    @SuppressLint(value={"VisibleForTests"})
    public synchronized void retryHttpConnectionWhenBackoffEnds() {
        Date currentTime = new Date(this.clock.currentTimeMillis());
        long retrySeconds = Math.max(0L, this.metadataClient.getRealtimeBackoffMetadata().getBackoffEndTime().getTime() - currentTime.getTime());
        this.makeRealtimeHttpConnection(retrySeconds);
    }

    private synchronized void makeRealtimeHttpConnection(long retryMilliseconds) {
        if (!this.canMakeHttpStreamConnection()) {
            return;
        }
        if (this.httpRetriesRemaining > 0) {
            --this.httpRetriesRemaining;
            this.scheduledExecutorService.schedule(new Runnable(){

                @Override
                public void run() {
                    ConfigRealtimeHttpClient.this.beginRealtimeHttpStream();
                }
            }, retryMilliseconds, TimeUnit.MILLISECONDS);
        } else if (!this.isInBackground) {
            this.propagateErrors(new FirebaseRemoteConfigClientException("Unable to connect to the server. Check your connection and try again.", FirebaseRemoteConfigException.Code.CONFIG_UPDATE_STREAM_ERROR));
        }
    }

    void setRealtimeBackgroundState(boolean backgroundState) {
        this.isInBackground = backgroundState;
    }

    private synchronized void resetRetryCount() {
        this.httpRetriesRemaining = 8;
    }

    private synchronized void setIsHttpConnectionRunning(boolean connectionRunning) {
        this.isHttpConnectionRunning = connectionRunning;
    }

    @SuppressLint(value={"VisibleForTests"})
    public synchronized ConfigAutoFetch startAutoFetch(HttpURLConnection httpURLConnection) {
        ConfigUpdateListener retryCallback = new ConfigUpdateListener(){

            @Override
            public void onUpdate(@NonNull ConfigUpdate configUpdate) {
            }

            @Override
            public void onError(@NonNull FirebaseRemoteConfigException error) {
                ConfigRealtimeHttpClient.this.enableBackoff();
                ConfigRealtimeHttpClient.this.propagateErrors(error);
            }
        };
        return new ConfigAutoFetch(httpURLConnection, this.configFetchHandler, this.activatedCache, this.listeners, retryCallback, this.scheduledExecutorService);
    }

    private boolean isStatusCodeRetryable(int statusCode) {
        return statusCode == 408 || statusCode == 429 || statusCode == 502 || statusCode == 503 || statusCode == 504;
    }

    private String parseForbiddenErrorResponseMessage(InputStream inputStream) {
        StringBuilder response;
        block3: {
            response = new StringBuilder();
            try {
                BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream));
                String message = "";
                while ((message = bufferedReader.readLine()) != null) {
                    response.append(message);
                }
            }
            catch (IOException ex) {
                if (response.length() != 0) break block3;
                return "Unable to connect to the server, access is forbidden. HTTP status code: 403";
            }
        }
        return response.toString();
    }

    @SuppressLint(value={"VisibleForTests", "DefaultLocale"})
    public void beginRealtimeHttpStream() {
        if (!this.canMakeHttpStreamConnection()) {
            return;
        }
        ConfigMetadataClient.RealtimeBackoffMetadata backoffMetadata = this.metadataClient.getRealtimeBackoffMetadata();
        Date currentTime = new Date(this.clock.currentTimeMillis());
        if (currentTime.before(backoffMetadata.getBackoffEndTime())) {
            this.retryHttpConnectionWhenBackoffEnds();
            return;
        }
        Task<HttpURLConnection> httpURLConnectionTask = this.createRealtimeConnection();
        Tasks.whenAllComplete(httpURLConnectionTask).continueWith(this.scheduledExecutorService, completedHttpUrlConnectionTask -> {
            boolean connectionFailed;
            Integer responseCode = null;
            HttpURLConnection httpURLConnection = null;
            try {
                if (!httpURLConnectionTask.isSuccessful()) {
                    throw new IOException(httpURLConnectionTask.getException());
                }
                this.setIsHttpConnectionRunning(true);
                httpURLConnection = (HttpURLConnection)httpURLConnectionTask.getResult();
                responseCode = httpURLConnection.getResponseCode();
                if (responseCode == 200) {
                    this.resetRetryCount();
                    this.metadataClient.resetRealtimeBackoff();
                    ConfigAutoFetch configAutoFetch = this.startAutoFetch(httpURLConnection);
                    configAutoFetch.listenForNotifications();
                }
                this.closeRealtimeHttpStream(httpURLConnection);
                this.setIsHttpConnectionRunning(false);
                boolean bl = connectionFailed = responseCode == null || this.isStatusCodeRetryable(responseCode);
            }
            catch (IOException e) {
                boolean connectionFailed2;
                try {
                    Log.d("FirebaseRemoteConfig", "Exception connecting to real-time RC backend. Retrying the connection...", e);
                    this.closeRealtimeHttpStream(httpURLConnection);
                    this.setIsHttpConnectionRunning(false);
                    boolean bl = connectionFailed2 = responseCode == null || this.isStatusCodeRetryable(responseCode);
                }
                catch (Throwable throwable) {
                    boolean connectionFailed3;
                    this.closeRealtimeHttpStream(httpURLConnection);
                    this.setIsHttpConnectionRunning(false);
                    boolean bl = connectionFailed3 = responseCode == null || this.isStatusCodeRetryable(responseCode);
                    if (connectionFailed3) {
                        this.updateBackoffMetadataWithLastFailedStreamConnectionTime(new Date(this.clock.currentTimeMillis()));
                    }
                    if (connectionFailed3 || responseCode == 200) {
                        this.retryHttpConnectionWhenBackoffEnds();
                    } else {
                        String errorMessage = String.format("Unable to connect to the server. Try again in a few minutes. HTTP status code: %d", responseCode);
                        if (responseCode == 403) {
                            errorMessage = this.parseForbiddenErrorResponseMessage(httpURLConnection.getErrorStream());
                        }
                        this.propagateErrors(new FirebaseRemoteConfigServerException((int)responseCode, errorMessage, FirebaseRemoteConfigException.Code.CONFIG_UPDATE_STREAM_ERROR));
                    }
                    throw throwable;
                }
                if (connectionFailed2) {
                    this.updateBackoffMetadataWithLastFailedStreamConnectionTime(new Date(this.clock.currentTimeMillis()));
                }
                if (connectionFailed2 || responseCode == 200) {
                    this.retryHttpConnectionWhenBackoffEnds();
                } else {
                    String errorMessage = String.format("Unable to connect to the server. Try again in a few minutes. HTTP status code: %d", responseCode);
                    if (responseCode == 403) {
                        errorMessage = this.parseForbiddenErrorResponseMessage(httpURLConnection.getErrorStream());
                    }
                    this.propagateErrors(new FirebaseRemoteConfigServerException((int)responseCode, errorMessage, FirebaseRemoteConfigException.Code.CONFIG_UPDATE_STREAM_ERROR));
                }
            }
            if (connectionFailed) {
                this.updateBackoffMetadataWithLastFailedStreamConnectionTime(new Date(this.clock.currentTimeMillis()));
            }
            if (connectionFailed || responseCode == 200) {
                this.retryHttpConnectionWhenBackoffEnds();
            } else {
                String errorMessage = String.format("Unable to connect to the server. Try again in a few minutes. HTTP status code: %d", responseCode);
                if (responseCode == 403) {
                    errorMessage = this.parseForbiddenErrorResponseMessage(httpURLConnection.getErrorStream());
                }
                this.propagateErrors(new FirebaseRemoteConfigServerException((int)responseCode, errorMessage, FirebaseRemoteConfigException.Code.CONFIG_UPDATE_STREAM_ERROR));
            }
            return Tasks.forResult(null);
        });
    }

    public void closeRealtimeHttpStream(HttpURLConnection httpURLConnection) {
        if (httpURLConnection != null) {
            httpURLConnection.disconnect();
            try {
                httpURLConnection.getInputStream().close();
                if (httpURLConnection.getErrorStream() != null) {
                    httpURLConnection.getErrorStream().close();
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }
}

