package io.bitsensor.plugins.java.http.filter.handler;

import io.bitsensor.lib.entity.proto.Error;
import io.bitsensor.plugins.java.core.BitSensor;
import org.springframework.stereotype.Component;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;

import static io.bitsensor.lib.entity.proto.GeneratedBy.PLUGIN;

/**
 * Handles cookies the client sent with this request
 *
 * @see <a href=https://www.ncsc.nl/actueel/whitepapers/ict-beveiligingsrichtlijnen-voor-webapplicaties.html>ICT-Beveiligingsrichtlijnen
 * voor Webapplicaties</a>
 */
@Component
public class CookieHandler implements RequestHandler {

    public static final String SECURE_ATTRIBUTE_NAME = "Secure";
    public static final String HTTP_ONLY_ATTRIBUTE_NAME = "HttpOnly";

    @Override
    public void handle(HttpServletRequest request, HttpServletResponse response) {

        if (request.getCookies() == null) {
            return;
        }

        for (Cookie cookie : request.getCookies()) {
            String cookieName = "cookie." + cookie.getName();
            BitSensor.addInput(cookieName, cookie.getValue());

            // checks compliance
            List<String> cookieCompliance = new ArrayList<>();
            if (!cookie.getSecure())
                cookieCompliance.add(SECURE_ATTRIBUTE_NAME);

            if (!cookie.isHttpOnly())
                cookieCompliance.add(HTTP_ONLY_ATTRIBUTE_NAME);

            if (cookieCompliance.isEmpty())
                return;

            BitSensor.addError(Error.newBuilder()
                    .setGeneratedby(PLUGIN)
                    .setCode(1)
                    .setDescription("Non-Compliance for " + cookieName + ": " + String.join(", ", cookieCompliance) + " flag(s) not set.")
                    .setType("compliance")
                    .build());
        }
    }

    @Override
    public ExecutionMoment getExecutionMoment() {
        return ExecutionMoment.PRE_HANDLE;
    }
}
