package com.sentray.kmmprotocolmodule.cryptor.baseCryptor

import com.sentray.kmmprotocolmodule.cryptor.common.RC4Encoder
import com.sentray.kmmprotocolmodule.cryptor.enumDefinition.CmdType
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.Cmd
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.ProtocolKey
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.ProtocolVersion
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolParser.parserUtil.value
import io.ktor.utils.io.core.*
import korlibs.crypto.encoding.Base64
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.decodeFromJsonElement
import kotlinx.serialization.json.jsonObject
import kotlin.jvm.JvmStatic

internal class L1DataCryptor {
    @Serializable
    data class L1Cmd(
        val sn: String,
        val seq: Long,
        val protocolVersion: Int,
        val cmdType: Int,
        val l1Data: String,
    )

    companion object {
        private val configuredJson = Json { ignoreUnknownKeys = true }

        @OptIn(ExperimentalSerializationApi::class)
        @JvmStatic
        fun encode(l1DataRaw: String): String? {
            return try {
                val jsonObject = Json.parseToJsonElement(l1DataRaw).jsonObject

                val sn = jsonObject.value<String>(ProtocolKey.Sn.raw) ?: return null
                val seq = jsonObject.value<Long>(ProtocolKey.Seq.raw) ?: return null
                val cmd = kotlin.run {
                    jsonObject.value<String>(ProtocolKey.Cmd.raw)?.let {
                        Cmd.fromRaw(it)
                    }
                } ?: return null

                val l1Key = getL1Key(sn, seq).encodeToByteArray()
//            val dataAfterRC4 = RC4.crypt(l1DataRaw, l1Key)
                val dataAfterRC4 =
                    RC4Encoder(l1Key).encode(l1DataRaw.encodeToByteArray(), 0, l1DataRaw.count())
                val l1Data = Base64.encode(dataAfterRC4)

                val protocolVersion = ProtocolVersion.V2.raw
                val cmdType = CmdType.getCmdType(cmd).raw

                val l1Cmd = L1Cmd(
                    sn,
                    seq,
                    protocolVersion,
                    cmdType,
                    l1Data
                )

                Json.encodeToString(l1Cmd)
            } catch (e: Exception) {
                e.printStackTrace()
                null
            }


        }

        //{"sn":"F0FE6B96A486F0FE6B96A486","seq":49930603,"protocolVersion":2,"cmdType":3,"l1Data":"EcKZwqRUFMKYVVscwpkIDcKtdifDkFfDqMO/wpxEG0XCpB9Qw7vDhMKvw6hya8K/RAnCrzI7w7HClhgNDE/CpXtCwrtQw6Fkw7rChylcwrERwo15wq3Ds8OqKklrFcKxwpI1w5k2w5pxw5Y+w5Z+w6Zxw4TDhWvCljUndm3DocOcwpHDt0oBwq7Drmw8w6XChTzDgsOgdMKwwpHDtsKwB8OMwrpYwqjDrsOEYsKowp4qwo/DkG53bFnChMOFw4nCoCHCnSgpP8OSGidiOjhuUgHDkBLCqMOrwofCtEFtHMORIcKdEMKkw7PCm3fCnsOKw73DgUPCszZPYMK7woosH8KTQMK/Py3DuGonDEzDr1zDokbDgsOeRsKXXMKsw7LCoS3CmBXChQBBwqzCl3fDhgvDiFPDnsOgw6liw6DDlB0zwp3DkV53w67DmcK3NcOyw68KwrtCw4oEwpxYw4otNWrCpEEZbsO2w4PCpylOIMO6wr4XKsOjI8Oywr9ZT8O0wrtdw4/Cj0LDs8KyScOeNl/CnMKHV8OMw5tmPsKzw5UPHsKvw4DCqsO4w5Buw6g9ZEA5RgsaYA=="}
        @OptIn(ExperimentalSerializationApi::class)
        @JvmStatic
        fun decode(jsonString: String): String? {
            return try {
                val l1Cmd = configuredJson.decodeFromString<L1Cmd>(jsonString)
                val l1DataWithRC4 = Base64.decode(l1Cmd.l1Data)
                val key = getL1Key(l1Cmd.sn, l1Cmd.seq).encodeToByteArray()
                val l1Data =
                    RC4Encoder(key).decode(l1DataWithRC4, 0, l1DataWithRC4.count()).decodeToString()
                l1Data
            } catch (e: Exception) {
                e.printStackTrace()
                null
            }
        }

        fun decode(jsonObject: JsonObject): String? {
            return try {
                val l1Cmd = configuredJson.decodeFromJsonElement<L1Cmd>(jsonObject)
                val l1DataWithRC4 = Base64.decode(l1Cmd.l1Data)
                val key = getL1Key(l1Cmd.sn, l1Cmd.seq).encodeToByteArray()
                val l1Data =
                    RC4Encoder(key).decode(l1DataWithRC4, 0, l1DataWithRC4.count()).decodeToString()
                l1Data
            } catch (e: Exception) {
                e.printStackTrace()
                null
            }
        }

        @JvmStatic
        private fun getL1Key(sn: String, seq: Long): String {
            val SN_COUNT = 5
            val SEQ_COUNT = 6
            val SEQ_MIN_VALUE = 1000000
            val SEQ_MAX_VALUE = 10000000
            val fixCodeByte =
                byteArrayOf(
                    '!'.code.toByte(), 'u'.code.toByte(), '_'.code.toByte(),
                    '#'.code.toByte(), '8'.code.toByte()
                )
            var mantissaSN: String
            val snLength = sn.length

            when {
                snLength > SN_COUNT -> {
                    mantissaSN = sn.substring(snLength - SN_COUNT)
                }
                snLength == SN_COUNT -> {
                    mantissaSN = sn
                }
                else -> {
                    mantissaSN = sn
                    val remain = SN_COUNT - snLength
                    for (i in 0 until remain) {
                        mantissaSN = "0$mantissaSN"
                    }
                }
            }
            var seqStr: String
            if (seq >= SEQ_MIN_VALUE) {
                seqStr = (seq % SEQ_MAX_VALUE).toString() + ""
                if (seqStr.length > SEQ_COUNT) {
                    seqStr = seqStr.substring(seqStr.length - SEQ_COUNT)
                }
            } else {
                seqStr = seq.toString() + ""
            }
            val remain = SEQ_COUNT - seqStr.length
            for (i in 0 until remain) {
                seqStr = "0$seqStr"
            }
            val snBytes: ByteArray = mantissaSN.toByteArray()
            val seqBytes: ByteArray =
                seqStr.toByteArray() //((System.currentTimeMillis()%1000000) + "").getBytes();
            val keyBytes = ByteArray(16)
            keyBytes[0] = fixCodeByte[0]
            keyBytes[1] = snBytes[0]
            keyBytes[2] = seqBytes[0]
            keyBytes[3] = fixCodeByte[1]
            keyBytes[4] = snBytes[1]
            keyBytes[5] = seqBytes[1]
            keyBytes[6] = fixCodeByte[2]
            keyBytes[7] = snBytes[2]
            keyBytes[8] = seqBytes[2]
            keyBytes[9] = fixCodeByte[3]
            keyBytes[10] = snBytes[3]
            keyBytes[11] = seqBytes[3]
            keyBytes[12] = fixCodeByte[4]
            keyBytes[13] = snBytes[4]
            keyBytes[14] = seqBytes[4]
            keyBytes[15] = seqBytes[5]
            return String(keyBytes)
        }
    }
}