package com.sentray.kmmprotocolmodule.sdk.command

import com.sentray.kmmprotocolmodule.sdk.parser.parserResult.ControllableApplianceInfo
import com.sentray.kmmprotocolmodule.sdk.parser.parserResult.DetectableApplianceStatus
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.ENetworkType
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.ProtocolVersion
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.WeekMask
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolCreator.CommandData
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolCreator.TimerCmdCreator
import com.sentray.kmmprotocolmodule.utilityToolKit.HexToolKit
import kotlinx.datetime.*
import kotlin.jvm.JvmStatic

//定时器相关命令
class TimerCmd {
    companion object {
        @JvmStatic
        fun getAllTimerCmd(
            sn: String,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            return TimerCmdCreator.getTimerV1(
                sn,
                protocolVersion = protocolVersion,
                networkType = networkType
            )
        }

        @Deprecated("TimerV1 已弃用")
        @JvmStatic
        fun addTimerV1Cmd(
            controllableApplianceInfo: ControllableApplianceInfo,
            timestamp: Long,
            enable: Boolean,
            action: Boolean,
            weekMaskFromUserTimezone: WeekMask,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            return TimerCmdCreator.addTimerV1(
                controllableApplianceInfo.sn,
                controllableApplianceInfo.devType,
                controllableApplianceInfo.applianceIndex,
                action,
                enable,
                timestamp,
                weekMaskFromUserTimezone,
                protocolVersion = protocolVersion,
                networkType = networkType
            )
        }

        @Deprecated("TimerV1 已弃用")
        @JvmStatic
        fun modifyTimerV1Cmd(
            timerIndex: Int,
            controllableApplianceInfo: ControllableApplianceInfo,
            timestamp: Long,
            enable: Boolean,
            action: Boolean,
            weekMaskFromUserTimezone: WeekMask,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val timestampNew = if (weekMaskFromUserTimezone.isEmpty()) {
                calculateNextValidTimestamp(timestamp)
            } else {
                timestamp
            }

            return TimerCmdCreator.modifyTimerV1(
                controllableApplianceInfo.sn,
                timerIndex,
                controllableApplianceInfo.devType,
                controllableApplianceInfo.applianceIndex,
                action,
                enable,
                timestampNew,
                weekMaskFromUserTimezone,
                protocolVersion = protocolVersion,
                networkType = networkType
            )
        }

        @Deprecated("TimerV1 已弃用")
        @JvmStatic
        fun deleteTimerV1Cmd(
            sn: String,
            timerIndex: Int,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            return TimerCmdCreator.deleteTimerV1(
                sn,
                timerIndex,
                protocolVersion = protocolVersion,
                networkType = networkType
            )
        }

        @JvmStatic
        fun addTimerV2Cmd(
            sn: String,
            userActionIndexList: List<Int>,
            detectableApplianceStatusList: List<DetectableApplianceStatus>,
            timestamp: Long,
            enable: Boolean,
            weekMaskFromUserTimezone: WeekMask,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val userActionMaskList = HexToolKit.indexListToBoolList(userActionIndexList)
            val triggerStatusList: List<Boolean>
            val triggerMaskList: List<Boolean>

            val triggerIndexList = mutableListOf<Int>()
            val triggerStatusIndexList = mutableListOf<Int>()
            detectableApplianceStatusList.forEach { detectableApplianceStatus ->
                triggerIndexList.add(detectableApplianceStatus.applianceIndex)
                if (detectableApplianceStatus.status) {
                    triggerStatusIndexList.add(detectableApplianceStatus.applianceIndex)
                }
            }

            triggerStatusList = HexToolKit.indexListToBoolList(triggerStatusIndexList)
            triggerMaskList = HexToolKit.indexListToBoolList(triggerIndexList)

            val timestampNew = if (weekMaskFromUserTimezone.isEmpty()) {
                calculateNextValidTimestamp(timestamp)
            } else {
                timestamp
            }

            return TimerCmdCreator.addTimerV2(
                sn,
                userActionMaskList,
                triggerStatusList,
                triggerMaskList,
                enable,
                timestampNew,
                weekMaskFromUserTimezone,
                protocolVersion = protocolVersion,
                networkType = networkType
            )
        }

        @JvmStatic
        fun modifyTimerV2Cmd(
            sn: String,
            timerIndex: Int,
            userActionIndexList: List<Int>,
            detectableApplianceStatusList: List<DetectableApplianceStatus>,
            timestamp: Long,
            enable: Boolean,
            weekMaskFromUserTimezone: WeekMask,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val userActionMaskList = HexToolKit.indexListToBoolList(userActionIndexList)
            val triggerStatusList: List<Boolean>
            val triggerMaskList: List<Boolean>

            val triggerIndexList = mutableListOf<Int>()
            val triggerStatusIndexList = mutableListOf<Int>()
            detectableApplianceStatusList.forEach { detectableApplianceStatus ->
                triggerIndexList.add(detectableApplianceStatus.applianceIndex)
                if (detectableApplianceStatus.status) {
                    triggerStatusIndexList.add(detectableApplianceStatus.applianceIndex)
                }
            }

            triggerStatusList = HexToolKit.indexListToBoolList(triggerStatusIndexList)
            triggerMaskList = HexToolKit.indexListToBoolList(triggerIndexList)

            val timestampNew = if (weekMaskFromUserTimezone.isEmpty()) {
                calculateNextValidTimestamp(timestamp)
            } else {
                timestamp
            }

            return TimerCmdCreator.modifyTimerV2(
                sn,
                timerIndex,
                userActionMaskList,
                triggerStatusList,
                triggerMaskList,
                enable,
                timestampNew,
                weekMaskFromUserTimezone,
                protocolVersion = protocolVersion,
                networkType = networkType
            )
        }

        @JvmStatic
        fun deleteTimerV2Cmd(
            sn: String,
            timerIndex: Int,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            return TimerCmdCreator.delTimerV2(
                sn,
                timerIndex,
                protocolVersion = protocolVersion,
                networkType = networkType
            )
        }

        //计算下一个有效 timestamp 的函数
        private fun calculateNextValidTimestamp(timestamp: Long): Long {
            val userTimezone = TimeZone.currentSystemDefault()

            val timerDate = Instant.fromEpochMilliseconds(timestamp * 1000)
            val timerDateTime = timerDate.toLocalDateTime(userTimezone)
            val currDateTime = Clock.System.now().toLocalDateTime(userTimezone)
            val timestampNew: Long

            //若当前时间比定时时间大，则重新生成定时时间
            if (currDateTime > timerDateTime) {
                val now = Clock.System.now()
                if (currDateTime.hour > timerDateTime.hour) {
                    now.plus(1, DateTimeUnit.DAY, userTimezone)
                } else if (currDateTime.hour == timerDateTime.hour) {
                    if (currDateTime.minute >= timerDateTime.minute) {
                        now.plus(1, DateTimeUnit.DAY, userTimezone)
                    }
                }
                val nowDateTime = now.toLocalDateTime(userTimezone)

                val newDateTime = LocalDateTime(
                    nowDateTime.year,
                    nowDateTime.month,
                    nowDateTime.dayOfMonth,
                    timerDateTime.hour,
                    timerDateTime.minute,
                    0,
                    0
                )
                timestampNew = newDateTime.toInstant(userTimezone).toEpochMilliseconds() / 1000
            } else {
                timestampNew = timestamp
            }

            return timestampNew
        }
    }
}