package com.sentray.kmmprotocolmodule.network.webSocket

import com.sentray.kmmprotocolmodule.httpClient
import com.sentray.kmmprotocolmodule.sdk.parser.Parser
import com.sentray.kmmprotocolmodule.utilityToolKit.genericClasses.KmmTimer
import com.sentray.kmmprotocolmodule.utilityToolKit.loggerKit.KMMLogger
import io.ktor.client.*
import io.ktor.client.plugins.websocket.*
import io.ktor.http.*
import io.ktor.websocket.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.channels.ClosedReceiveChannelException
import kotlinx.coroutines.channels.SendChannel
import kotlinx.coroutines.launch
import kotlin.time.Duration.Companion.seconds

//https://ktor.io/docs/getting-started-ktor-client-chat.html#wire-it-togethers
//多平台WebSocket实现 : https://ktor.io/docs/http-client-engines.html#mpp-config
internal class WebSocketClient(
    private val host: String,
    private val port: Int,
    private val userID: Int,
    private val accessToken: String,
    private val decode: Boolean
) {
    internal enum class EWebSocketStatus {
        Connected,
        Disconnected,
        Connecting,
    }

    private var client: HttpClient? = null
    private var reConnectTimer: KmmTimer? = null
    private val reConnectCheckDuration = 3.seconds
    private var status = EWebSocketStatus.Disconnected
    private val parser = Parser()

    private var sendChannel: SendChannel<Frame>? = null

    suspend fun startConnect(): EWebSocketStatus {
        disconnect()
        status = EWebSocketStatus.Connecting
        client = httpClient {
            install(WebSockets) {
            }
        }
        CoroutineScope(Dispatchers.Default).launch {
            client?.let { clientValid ->
                clientValid.wss(
                    method = HttpMethod.Get,
                    host = host,
                    port = port,
                    path = "?userID=$userID&accessToken=$accessToken&decode=$decode",
                ) {
                    sendChannel = outgoing

                    status = EWebSocketStatus.Connected
                    startReconnectTimer()

                    //连接成功后，等待接收数据
                    try {
                        while (client != null) {
                            val message = (incoming.receive() as Frame.Text).readText()
                            KMMLogger.d("WebSocket Receive Message : $message")
                            parser.parse(message).let { resultList ->
                                resultList.forEach {
                                    KMMLogger.d("WebSocket Parse : ${it.stringify()}")
                                }
                            }
                        }
                    } catch (e: ClosedReceiveChannelException) {
                        KMMLogger.d("onClose ${closeReason.await()}")
                        status = EWebSocketStatus.Disconnected
                        e.printStackTrace()
                    } catch (e: Throwable) {
                        KMMLogger.d("onError ${closeReason.await()}")
                        status = EWebSocketStatus.Disconnected
                        e.printStackTrace()
                    }
                }
            }
        }
        return status
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    suspend fun sendData(message: String) {
        sendChannel?.let { sendChannelValid ->
            if (!sendChannelValid.isClosedForSend) {
                sendChannelValid.send(Frame.Text(message))
            }
        }
    }


    fun isConnected(): Boolean {
        return status == EWebSocketStatus.Connected
    }

    fun disconnectManually() {
        disconnect()
        stopReconnectTimer()
    }

    private fun disconnect() {
        sendChannel?.close()
        sendChannel = null

        client?.close()
        client = null

        status = EWebSocketStatus.Disconnected
    }

    private fun startReconnectTimer() {
        stopReconnectTimer()
        if (reConnectTimer == null) {
            reConnectTimer = KmmTimer().also {
                it.startScheduleTimer(reConnectCheckDuration) {
                    if (status == EWebSocketStatus.Disconnected) {
                        CoroutineScope(Dispatchers.Default).launch {
                            KMMLogger.d("Websocket 链接断开，尝试重连")
                            startConnect()
                            KMMLogger.d("Websocket 重连结果：$status")
                        }
                    }
                    KMMLogger.d("checking webSocket connection : ")
                }
            }
        }
    }

    private fun stopReconnectTimer() {
        reConnectTimer?.stopScheduleTimer()
        reConnectTimer = null
    }
}