package com.sentray.kmmprotocolmodule.task.commandSender

import com.sentray.kmmprotocolmodule.network.tcp.TcpClientManager
import com.sentray.kmmprotocolmodule.network.webSocket.WebSocketClientManager
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.ENetworkType
import com.sentray.kmmprotocolmodule.utilityToolKit.genericClasses.KmmTimer
import com.sentray.kmmprotocolmodule.utilityToolKit.genericClasses.MutableLinkList
import com.sentray.kmmprotocolmodule.utilityToolKit.genericClasses.mutableLinkListOf
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlin.time.Duration.Companion.milliseconds

@OptIn(ExperimentalCoroutinesApi::class)
internal object SchedulePoster {
    enum class ECmdPriority {
        Low,
        High
    }

    enum class EQueueType {
        Queue100ms,
        Queue600ms
    }

    private val timer100ms = KmmTimer()
    private val timer600ms = KmmTimer()

    //保证 cmdCache 在同一个协程读写
    private val cmdCacheQueueRWCoroutineScope = newSingleThreadContext("cmdCacheRWContext")
    private val cmdCache100ms = HashMap<String, CmdCacheQueue>()
    private val cmdCache600ms = HashMap<String, CmdCacheQueue>()

    suspend fun pushToQueue(posterData: PosterData, queueType: EQueueType, priority: ECmdPriority) {
        CoroutineScope(cmdCacheQueueRWCoroutineScope).launch {
            when (queueType) {
                EQueueType.Queue100ms -> {
                    cmdCache100ms[posterData.mac] ?: run {
                        cmdCache100ms[posterData.mac] = CmdCacheQueue()
                    }
                    when (priority) {
                        ECmdPriority.Low -> {
                            cmdCache100ms[posterData.mac]?.push(posterData)
                        }
                        ECmdPriority.High -> {
                            cmdCache100ms[posterData.mac]?.insertFirst(posterData)
                        }
                    }
                }
                EQueueType.Queue600ms -> {
                    cmdCache600ms[posterData.mac] ?: run {
                        cmdCache600ms[posterData.mac] = CmdCacheQueue()
                    }
                    when (priority) {
                        ECmdPriority.Low -> {
                            cmdCache600ms[posterData.mac]?.push(posterData)
                        }
                        ECmdPriority.High -> {
                            cmdCache600ms[posterData.mac]?.insertFirst(posterData)
                        }
                    }
                }
            }
        }
    }

    fun start() {
        timer100ms.startScheduleTimer(100.milliseconds, 0.milliseconds) {
            CoroutineScope(cmdCacheQueueRWCoroutineScope).launch {
                cmdCache100ms.forEach {
                    it.value.pop()?.let { posterData ->
                        postDataHandler(posterData)
                    }
                }
            }
        }
        timer600ms.startScheduleTimer(600.milliseconds, 0.milliseconds) {
            CoroutineScope(cmdCacheQueueRWCoroutineScope).launch {
                cmdCache600ms.forEach {
                    it.value.pop()?.let { posterData ->
                        postDataHandler(posterData)
                    }
                }
            }
        }
    }

    fun stop() {
        CoroutineScope(cmdCacheQueueRWCoroutineScope).launch {
            cmdCache100ms.clear()
            cmdCache600ms.clear()
        }
        timer100ms.stopScheduleTimer()
        timer600ms.stopScheduleTimer()
    }

    private fun postDataHandler(posterData: PosterData) {
        when (posterData.networkType) {
            ENetworkType.Lan -> {
                TcpClientManager.sendDataToController(posterData.mac, posterData.message)
            }
            ENetworkType.Cloud -> {
                WebSocketClientManager.send(posterData.message)
            }
        }
    }

    private class CmdCacheQueue {
        private var readWriteLock = Mutex()
        private var linkList: MutableLinkList<PosterData> = mutableLinkListOf()

        suspend fun push(posterData: PosterData) {
            readWriteLock.withLock {
                linkList.addLast(posterData)
            }
        }

        suspend fun insertFirst(posterData: PosterData) {
            readWriteLock.withLock {
                linkList.addFirst(posterData)
            }
        }

        suspend fun pop(): PosterData? {
            readWriteLock.withLock {
                return linkList.getFirst()
            }
        }
    }
}