package com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent

import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeSet
import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeUtil
import com.sentray.kmmprotocolmodule.utilityToolKit.HexToolKit
import kotlinx.datetime.*
import kotlinx.serialization.Serializable
import kotlin.math.abs

/*
    Timer V1 raw string 结构说明 : "1a,posix,7f,1,5,ffffffff,ffffffff" , "......"
    内部格式如下：
    	1a : timer index, 表示50个timer中的第几个timer.
    	posix : 定时信息，以GMT时间为依据，得到从1970.1.1 00:00:00至今的秒数。
        	若是循环定时：则年月日为设置闹钟时的日期，时分秒为用户设定的时分，WiLink-mom在处理循环定时事件时，会自动忽略年月日。
        	若是一次性定时：APP直接将用户设置的年月日时分信息转换成posix时间.
    	7f : week mask，00(非循环定时)；7f(7天全部定时)；XX(星期X循环)，如13，转二进制为001 0011，代表周日，一，四循环。
    	1 : 1:timer enable		0:timer disable
    	5 : device type (int类型，具体参考“从机类型编号表”)
    	ffffffff : actionMask 用于表示当定时时间到后，哪个开关做什么动作，与下面的rlyMask配合使用。
    	ffffffff : rlyMask 用于表示当定时时间 ，哪个开关需要动作，与上面的actionMask配合使用。使用时，首先检查rlyMask，若bit为1，则该bit对应的开关需要执行动作，至于是执行开还是关动作，就看actionMask上对应的bit，1为闭合，0为断开；
     */
@Serializable
internal class TimerParaV1 {
    var timerBasePara: TimerBasePara = TimerBasePara()
    var actionMask: List<Boolean> = listOf()
    var rlyMask: List<Boolean> = listOf()

   
    fun initialFromRaw(timerParaV1String: String) {
        //1. 将字符串分组，分隔符为 ","
        val timerParaList = timerParaV1String.split(",")
        if (timerParaList.count() == 7) {
            //2. 解析数据
            //timer index
            HexToolKit.hexStringToULong(timerParaList[0])?.let {
                this.timerBasePara.timerIndex = it.toInt()
            }
            //posix
            HexToolKit.hexStringToULong(timerParaList[1])?.let {
                this.timerBasePara.posix = it.toLong()
            }
            //week mask : 中控内时区为 GMT+0，APP需根据用户所在时区作转换处理
            HexToolKit.hexStringToBoolList(timerParaList[2], 7)?.let { boolList ->
                this.timerBasePara.setWeekMaskUTC0FromProtocol(boolList)
            }

            //enable
            HexToolKit.hexStringToULong(timerParaList[3])?.let {
                this.timerBasePara.enable = (it > 0UL)
            }

            //device type
            HexToolKit.hexStringToULong(timerParaList[4])?.let {
                DeviceTypeUtil.findDeviceTypeString(it.toInt())?.let { deviceTypeString ->
                    DeviceTypeSet.fromRaw(deviceTypeString)?.let { deviceTypeSet ->
                        this.timerBasePara.devType = deviceTypeSet
                    }
                }
            }

            //actionMask
            HexToolKit.hexStringToBoolList(timerParaList[5])?.let {
                this.actionMask = it
            }

            //rlyMask
            HexToolKit.hexStringToBoolList(timerParaList[6])?.let {
                this.rlyMask = it
            }
        }
    }

    fun toTimerParaV1String(): String {
        var timerString = ""

        //1. timerIndex
        timerString += "${this.timerBasePara.timerIndex.toString(16).uppercase()},"

        //2. posix time
        timerString += "${this.timerBasePara.posix.toString(16).uppercase()},"

        //3. week mask
        timerString += "${this.timerBasePara.getWeekMaskForUTC0().toHexString()},"

        //4. enable bit
        val enableBitString = when (timerBasePara.enable) {
            true -> "1"
            false -> "0"
        }
        timerString += "${enableBitString},"

        //5. deviceType
        DeviceTypeUtil.findDeviceTypeNumber(timerBasePara.devType.raw)?.let {
            timerString += "${it.toString(16).uppercase()},"
        }

        //6. action mask
        timerString += "${HexToolKit.boolListToHexString(actionMask)},"

        //7. relay mask
        timerString += HexToolKit.boolListToHexString(rlyMask)

        return timerString
    }

    @Serializable
    class TimerBasePara {
        var timerIndex: Int = 0
        var posix: Long = 0
        private var weekMaskForUTC0: WeekMask = WeekMask() //week mask 需要根据用户所处时区进行调整

        //        private var weekMaskForUserTimeZone: WeekMask = WeekMask()
        var enable: Boolean = false
        var devType: DeviceTypeSet = DeviceTypeSet.Unknown

        fun setWeekMaskUTC0FromProtocol(boolList: List<Boolean>) {
            weekMaskForUTC0.setFromBoolList(boolList)
        }

        fun getWeekMaskForUTC0(): WeekMask {
            return weekMaskForUTC0
        }

        fun setWeekMaskForUTC0(weekMaskFromUserTimeZone: WeekMask) {
            val userTimeZone = TimeZone.currentSystemDefault()
            val userTimeZoneOffset = userTimeZone.offsetAt(Clock.System.now())
            val timeZoneInterval = userTimeZoneOffset.totalSeconds / 3600

            val timerDate = Instant.fromEpochMilliseconds(this.posix * 1000)
            val timerHourInUserTimeZone =
                timerDate.toLocalDateTime(userTimeZone).hour  //定时器中的用户时区的 hour 数值

//            KMMLogger.d("setWeekMaskForUTC0 测试数据如下 : ")
//            KMMLogger.d("userTimeZone = $userTimeZone, userTimeZoneOffset = $userTimeZoneOffset, timeZoneInterval = $timeZoneInterval, timerHourInUserTimeZone = $timerHourInUserTimeZone ")

            //计算原理：
            // 1. 假设用户时区为 UTC+8, 中控时区为 UTC+0
            // 1.1 假设定时的 hour 为 UTC+8 时区的早上 2 时（小于 8 时），则对于 UTC+0 来说，定时的 hour 是上一天的 18 时，
            // 所以 UTC+8 的 weekMask 需要左移一位转换成 UTC+0 时区的 weekMask
            // 2. 假设用户时区为 UTC-6, 中控时区为 UTC+0
            // 2.1 假设定时的 hour 为 UTC-6 时区的晚上 22 点（(24 - 22) 小于 6 小时），则对于 UTC+0 来说，定时的 hour 是下一天的早上 4 点，
            // 所以 UTC-6 的 weekMask 需要右移一位转换成 UTC+0 时区的 weekMask
            val weekMaskForUTC0New = WeekMask()
            when {
                //若 userTimeZone 为 UTC+ 时区
                timeZoneInterval > 0 -> {
                    //若定时hour小于interval，则所有 weekMaskList 左移一位
                    if (timerHourInUserTimeZone < abs(timeZoneInterval)) {
                        WeekMaskIndex.values().forEach {
                            if (weekMaskFromUserTimeZone.getWeekMaskBit(it)) {
                                if (it == WeekMaskIndex.Sunday) {
                                    weekMaskForUTC0New.setWeekMaskBit(WeekMaskIndex.Saturday, true)
                                } else {
                                    WeekMaskIndex.fromRaw(it.raw - 1)?.let { newWeekDay ->
                                        weekMaskForUTC0New.setWeekMaskBit(newWeekDay, true)
                                    }
                                }
                            }
                        }
                    } else {
                        weekMaskForUTC0New.setFromWeekMask(weekMaskFromUserTimeZone)
                    }
                }
                //若 userTimeZone 为 UTC- 时区
                timeZoneInterval < 0 -> {
                    //若定时hour小于interval，则所有weekday + 1
                    if ((24 - timerHourInUserTimeZone) < abs(timeZoneInterval)) {
                        WeekMaskIndex.values().forEach {
                            if (weekMaskFromUserTimeZone.getWeekMaskBit(it)) {
                                if (it == WeekMaskIndex.Saturday) {
                                    weekMaskForUTC0New.setWeekMaskBit(WeekMaskIndex.Sunday, true)
                                } else {
                                    WeekMaskIndex.fromRaw(it.raw + 1)?.let { newWeekDay ->
                                        weekMaskForUTC0New.setWeekMaskBit(newWeekDay, true)
                                    }
                                }
                            }
                        }
                    } else {
                        weekMaskForUTC0New.setFromWeekMask(weekMaskFromUserTimeZone)
                    }
                }
                else -> {
                    weekMaskForUTC0New.setFromWeekMask(weekMaskFromUserTimeZone)
                }
            }
            weekMaskForUTC0.setFromWeekMask(weekMaskForUTC0New)
        }

        fun getWeekMaskForUserTimeZone(): WeekMask {
            //1. 计算新时区与原时区之间的差
            val userTimeZone = TimeZone.currentSystemDefault()
            val userTimeZoneOffset = userTimeZone.offsetAt(Clock.System.now())

            val timeZoneInterval = userTimeZoneOffset.totalSeconds / 3600

            val timerDate = Instant.fromEpochMilliseconds(this.posix * 1000)
            val timerHourInUserTimeZone =
                timerDate.toLocalDateTime(userTimeZone).hour  //定时器中的用户时区的 hour 数值

            //计算原理：
            // 1. 假设用户时区为 UTC+8, 中控时区为 UTC+0
            // 1.1 假设定时的 hour 为 UTC+8 时区的早上 2 时（小于 8 时），则对于 UTC+0 来说，定时的 hour 是上一天的 18 时，
            // 所以 UTC+0 的 weekMask 需要右移一位转换成用户时区的 weekMask
            // 2. 假设用户时区为 UTC-6, 中控时区为 UTC+0
            // 2.1 假设定时的 hour 为 UTC-6 时区的晚上 22 点（(24 - 22) 小于 6 小时），则对于 UTC+0 来说，定时的 hour 是下一天的早上 4 点，
            // 所以 UTC+0 的 weekMask 需要左移一位转换成用户时区的 weekMask
            val weekMaskNew = WeekMask()
            when {
                //若 userTimeZone 为 UTC+ 时区
                timeZoneInterval > 0 -> {
                    //若定时hour小于interval，则 weekMaskList 右移一位
                    if (timerHourInUserTimeZone < abs(timeZoneInterval)) {
                        WeekMaskIndex.values().forEach {
                            if (weekMaskForUTC0.getWeekMaskBit(it)) {
                                if (it == WeekMaskIndex.Saturday) {
                                    weekMaskNew.setWeekMaskBit(WeekMaskIndex.Sunday, true)
                                } else {
                                    WeekMaskIndex.fromRaw(it.raw + 1)?.let { newWeekDay ->
                                        weekMaskNew.setWeekMaskBit(newWeekDay, true)
                                    }
                                }
                            }
                        }
                    } else {
                        weekMaskNew.setFromWeekMask(weekMaskForUTC0)
                    }
                }
                //若 userTimeZone 为 UTC- 时区
                timeZoneInterval < 0 -> {
                    //若定时hour小于interval，则 weekMaskList 左移一位
                    if ((24 - timerHourInUserTimeZone) < abs(timeZoneInterval)) {
                        WeekMaskIndex.values().forEach {
                            if (weekMaskForUTC0.getWeekMaskBit(it)) {
                                if (it == WeekMaskIndex.Sunday) {
                                    weekMaskNew.setWeekMaskBit(WeekMaskIndex.Saturday, true)
                                } else {
                                    WeekMaskIndex.fromRaw(it.raw - 1)?.let { newWeekDay ->
                                        weekMaskNew.setWeekMaskBit(newWeekDay, true)
                                    }
                                }
                            }
                        }
                    } else {
                        weekMaskNew.setFromWeekMask(weekMaskForUTC0)
                    }

                }
                else -> {
                    weekMaskNew.setFromWeekMask(weekMaskForUTC0)
                }
            }
//            KMMLogger.d("before set weekMaskForUserTimeZone")
//            weekMaskForUserTimeZone.initialFromWeekMask(weekMaskNew)
//            KMMLogger.d("after set weekMaskForUserTimeZone")
            return weekMaskNew
        }
    }
}