package com.sentray.kmmprotocolmodule.sdk.parser.parserResult

//import com.sentray.kmmprotocolmodule.freeze
import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeClassify
import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeSet
import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeUtil
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.ProtocolContent
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.OnOffStatus
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.DevicePara
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.FirmwareVersion
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.ParaStructure
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolCreator.Util
import kotlinx.serialization.Serializable

@Serializable
class ControllableDeviceInfo {
    private var sn: String = ""
    var installedControllableDeviceTypeList: List<DeviceTypeSet>? = null
    var updatedControllableApplianceInfoList: List<ControllableApplianceInfo>? = null
    var updatedAirQualityInfoList: List<ControllableApplianceInfo.AirQualityInfo>? = null
    var updatedAirQualityPlusInfoList: List<ControllableApplianceInfo.AirQualityPlusInfo>? = null

    private var applianceInfoMap = mutableMapOf<String, ControllableApplianceInfo>()

    internal constructor(
        sn: String,
        outputDevContent: ProtocolContent.OutputDevContent,
    ) {
        outputDevContent.outputDeviceInstalled?.let { outputDeviceInstalledList ->
            this.sn = sn
            val installedList = mutableListOf<DeviceTypeSet>()
            for (devTypeNum in 0 until outputDeviceInstalledList.count()) {
                if (outputDeviceInstalledList[devTypeNum]) {
                    DeviceTypeUtil.findDeviceTypeString(devTypeNum)?.let { devTypeString ->
                        DeviceTypeSet.fromRaw(devTypeString)?.let { deviceTypeSet ->
                            installedList.add(deviceTypeSet)
                        }
                    }
                }
            }
            installedControllableDeviceTypeList = installedList.toList()
        }

        outputDevContent.devParaMap?.let { devParaMap ->
            if (devParaMap.isNotEmpty()) {
                this.sn = sn
            }
            devParaMap.forEach {
                DeviceTypeSet.fromRaw(it.key.raw)?.let { deviceTypeSet ->
                    val devicePara = it.value

                    //handle rfInstall
                    rfInstallHandle(deviceTypeSet, devicePara)

                    //TODO: handle index in rfLearnAck V2，貌似没用

                    //handle rlyMask & rlyStatus
                    rlyMaskAndStatusHandle(deviceTypeSet, devicePara)

                    //handle onlineStatus
                    onlineStatusHandle(deviceTypeSet, devicePara)

                    //handle ledMask & ledStatus
                    ledStatusHandle(deviceTypeSet, devicePara)

                    //handle onOffStatus & onOffStatusMask
                    onOffStatusHandle(deviceTypeSet, devicePara)

                    //handle brightPercent
                    brightPercentHandle(deviceTypeSet, devicePara)

                    //handle confInfo
                    dimmerConfigureInfoHandle(deviceTypeSet, devicePara)

                    //handle para & paraMask
                    paraAndParaMaskHandle(deviceTypeSet, devicePara)

                    //handle firmware version
                    firmwareVersionHandle(deviceTypeSet, devicePara)
                }
            }
            if (applianceInfoMap.isNotEmpty()) {
                updatedControllableApplianceInfoList = applianceInfoMap.values.toList()
            }
        }
    }

//    fun freezeSelf() {
//        freeze(this)
//    }

    private fun rfInstallHandle(deviceTypeSet: DeviceTypeSet, devicePara: DevicePara) {

        val rfInstallList = if (DeviceTypeClassify.isWifiOnlyDevice(deviceTypeSet)) {
            listOf(true)
        } else {
            devicePara.rfInstall
        }

        rfInstallList?.let {
            if (DeviceTypeClassify.isGatewayDevice(deviceTypeSet)) {
                for (index in 0 until it.count()) {
                    if (index == 0) {
                        //中控设备生成
                        for (applianceIndex in index * 3 until index * 3 + 3) {
                            createApplianceInfo(deviceTypeSet, index, applianceIndex)
                        }
                    } else {
                        if (it[index]) {
                            for (applianceIndex in index * 3 until index * 3 + 3) {
                                createApplianceInfo(deviceTypeSet, index, applianceIndex)
                            }
                        }
                    }
                }
            } else {
                for (index in 0 until it.count()) {
                    if (it[index]) {
                        val applianceAmount = Util.getApplianceAmount(deviceTypeSet)
                        for (applianceIndex in index * applianceAmount until index * applianceAmount + applianceAmount) {
                            createApplianceInfo(deviceTypeSet, index, applianceIndex)
                        }
                    }
                }
            }
        }
    }

    private fun rlyMaskAndStatusHandle(deviceTypeSet: DeviceTypeSet, devicePara: DevicePara) {
        var rlyStatus = devicePara.rlyStatus
        var rlyMask = devicePara.rlyMask

        if (DeviceTypeClassify.isWifiOnlyDevice(deviceTypeSet)) {
            //纯 wifi 设备的 rlyMask 和 rlyStatus 可能需要处理
            rlyMask = rlyMask ?: listOf(true)
            rlyStatus = rlyStatus ?: listOf(true)
        }

        if (rlyMask == null && rlyStatus != null) {
            //for mom's getAck, only update mom's rlyStatus.
            rlyMask = listOf(true, true, true)
        } else if (rlyMask != null && rlyStatus == null) {
            rlyStatus = rlyMask
        }

        rlyMask ?: return
        rlyStatus ?: return

        for (applianceIndex in 0 until rlyMask.count()) {
            if (rlyMask[applianceIndex]) {
                val applianceInfo = createApplianceInfo(
                    deviceTypeSet,
                    Util.getDeviceIndex(deviceTypeSet, applianceIndex),
                    applianceIndex
                )

                if (applianceIndex < rlyStatus.count()) {
                    applianceInfo.status = rlyStatus[applianceIndex]
                }
                applianceInfo.onlineStatus = true
            }
        }
    }

    private fun onlineStatusHandle(deviceTypeSet: DeviceTypeSet, devicePara: DevicePara) {
        //TODO : 貌似意义不大，考虑去除
        devicePara.onlineStatus?.let { onlineStatusValid ->
            val applianceAmount = Util.getApplianceAmount(deviceTypeSet)
            for (deviceIndex in 0 until onlineStatusValid.count()) {
                for (applianceIndex in deviceIndex * applianceAmount until deviceIndex * applianceAmount + applianceAmount) {
                    val key = ControllableApplianceInfo.createKey(sn, deviceTypeSet, applianceIndex)
                    applianceInfoMap[key]?.let {
                        it.onlineStatus = onlineStatusValid[deviceIndex]
                    }
                }
            }
        }
    }

    private fun ledStatusHandle(deviceTypeSet: DeviceTypeSet, devicePara: DevicePara) {
        val ledStatus = devicePara.ledStatus ?: return
        val ledMask = devicePara.ledMask ?: return

        for (applianceIndex in 0 until ledMask.count()) {
            if (ledMask[applianceIndex] && applianceIndex < ledStatus.count()) {
                val key = ControllableApplianceInfo.createKey(sn, deviceTypeSet, applianceIndex)
                applianceInfoMap[key]?.let {
                    it.ledStatus = ledStatus[applianceIndex]
                }
            }
        }
    }

    private fun onOffStatusHandle(deviceTypeSet: DeviceTypeSet, devicePara: DevicePara) {
        val onOffStatus = devicePara.onOffStatus
        var onOffStatusMask = devicePara.onOffRlyMask

        if (onOffStatusMask == null && onOffStatus != null) {
            onOffStatusMask = List(onOffStatus.count()) { true }
        }

        onOffStatus ?: return
        onOffStatusMask ?: return

        for (applianceIndex in 0 until onOffStatusMask.count()) {
            if (onOffStatusMask[applianceIndex] && applianceIndex < onOffStatus.count()) {
                val key = ControllableApplianceInfo.createKey(sn, deviceTypeSet, applianceIndex)
                applianceInfoMap[key]?.let {
                    it.powerOnStatus = onOffStatus[applianceIndex]
                }
            }
        }
    }


    private fun brightPercentHandle(deviceTypeSet: DeviceTypeSet, devicePara: DevicePara) {
        val valueList = devicePara.brightPercent ?: return
        val mask = devicePara.brightPercentMask ?: return

        var valueReadIndex = 0
        for (applianceIndex in 0 until mask.count()) {
            if (mask[applianceIndex] && valueReadIndex < valueList.count()) {
                val key = ControllableApplianceInfo.createKey(sn, deviceTypeSet, applianceIndex)
                applianceInfoMap[key]?.let {
                    it.para = ParaStructure(valueList[valueReadIndex].toULong())
                }
                valueReadIndex++
            }
        }
    }


    private fun dimmerConfigureInfoHandle(deviceTypeSet: DeviceTypeSet, devicePara: DevicePara) {
        val confMask = devicePara.confMask ?: return

        var valueReadIndex = 0
        for (applianceIndex in 0 until confMask.count()) {
            if (confMask[applianceIndex]) {
                val key = ControllableApplianceInfo.createKey(sn, deviceTypeSet, applianceIndex)
                applianceInfoMap[key]?.let { applianceInfo ->
                    val turnOnModeRaw: Int = devicePara.turnOnMode?.let {
                        if (valueReadIndex < it.count()) {
                            it[valueReadIndex]
                        } else {
                            0
                        }
                    } ?: 0
                    val turnOnBrightPercentRaw = devicePara.turnOnBrightPercent?.let {
                        if (valueReadIndex < it.count()) {
                            it[valueReadIndex]
                        } else {
                            0
                        }
                    } ?: 0

                    val minBrightPercentRaw = devicePara.minBrightPercent?.let {
                        if (valueReadIndex < it.count()) {
                            it[valueReadIndex]
                        } else {
                            0
                        }
                    } ?: 0

                    val brightAdjSpeedRaw = devicePara.brightAdjSpeed?.let {
                        if (valueReadIndex < it.count()) {
                            it[valueReadIndex]
                        } else {
                            0
                        }
                    } ?: 0
                    val onOffBrightAdjSpeedRaw = devicePara.onOffBrightAdjSpeed?.let {
                        if (valueReadIndex < it.count()) {
                            it[valueReadIndex]
                        } else {
                            0
                        }
                    } ?: 0
                    val slowlyOnBrightAdjSpeedRaw = devicePara.slowlyOnBrightAdjSpeed?.let {
                        if (valueReadIndex < it.count()) {
                            it[valueReadIndex]
                        } else {
                            0
                        }
                    } ?: 0
                    val slowlyOffBrightAdjSpeedRaw = devicePara.slowlyOffBrightAdjSpeed?.let {
                        if (valueReadIndex < it.count()) {
                            it[valueReadIndex]
                        } else {
                            0
                        }
                    } ?: 0
                    val confInfo = ControllableApplianceInfo.ConfInfo(
                        deviceTypeSet,
                        turnOnModeRaw,
                        turnOnBrightPercentRaw,
                        minBrightPercentRaw,
                        brightAdjSpeedRaw,
                        onOffBrightAdjSpeedRaw,
                        slowlyOnBrightAdjSpeedRaw,
                        slowlyOffBrightAdjSpeedRaw,
                        devicePara.turnOnBrightPercent ?: listOf(0)
                    )
                    applianceInfo.confInfo = confInfo

                    devicePara.powerOnStatus?.let {
                        if (valueReadIndex < it.count()) {
                            applianceInfo.powerOnStatus = it[valueReadIndex]
                        }
                    }
                }
                valueReadIndex++
            }
        }
    }

    private fun paraAndParaMaskHandle(deviceTypeSet: DeviceTypeSet, devicePara: DevicePara) {
        var paraMask = devicePara.paraMask
        var paraList = devicePara.para

        //若是电能统计设备，则对应的关键字是 power
        if (DeviceTypeClassify.isPowerStatisticsDevice(deviceTypeSet)) {
            paraList = devicePara.power
        }

        if (paraMask == null) {
            //特殊处理，ENV_MONITOR_W 没有 paraMask 字段
            if (DeviceTypeClassify.isEnvMonitorDevice(deviceTypeSet)) {
                paraMask = listOf(true)
            }
        }

        paraMask?.let { paraMaskValid ->
            var paraReadIndex = 0
            for (applianceIndex in 0 until paraMaskValid.count()) {
                if (paraMaskValid[applianceIndex]) {
                    val key = ControllableApplianceInfo.createKey(sn, deviceTypeSet, applianceIndex)
                    applianceInfoMap[key]?.let { applianceInfo ->
                        paraList?.let { paraListValid ->
                            if (DeviceTypeClassify.isEnvMonitorDevice(deviceTypeSet)) {
                                applianceInfo.onlineStatus = true
                                envParaHandle(deviceTypeSet, applianceIndex, paraListValid)
                            } else if (DeviceTypeClassify.isRGBLampDevice(deviceTypeSet) ||
                                DeviceTypeClassify.isRGBWLampDevice(deviceTypeSet)
                            ) {
                                applianceInfo.para = rbgDeviceParaHandle(paraListValid)
                            } else if (DeviceTypeClassify.isIRDevice(deviceTypeSet)) {
                                applianceInfo.irInfoList =
                                    irDeviceParaHandle(deviceTypeSet, applianceIndex, paraListValid)
                            } else if (DeviceTypeClassify.isPowerStatisticsDevice(deviceTypeSet)) {
                                if (paraReadIndex < paraListValid.count()) {
                                    applianceInfo.para =
                                        ParaStructure((paraListValid[paraReadIndex] / 10).toULong())
                                }
                            } else {
                                if (paraReadIndex < paraListValid.count()) {
                                    applianceInfo.para =
                                        ParaStructure(paraListValid[paraReadIndex].toULong())
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    private fun firmwareVersionHandle(deviceTypeSet: DeviceTypeSet, devicePara: DevicePara) {
        val mask = devicePara.rlyMask ?: devicePara.paraMask ?: return
        val firmwareVersion = devicePara.ver ?: return

        for (applianceIndex in 0 until mask.count()) {
            if (mask[applianceIndex]) {
                val key = ControllableApplianceInfo.createKey(sn, deviceTypeSet, applianceIndex)
                applianceInfoMap[key]?.let { applianceInfo ->
                    applianceInfo.fwVersion = firmwareVersion
                }
            }
        }
    }

    private fun createApplianceInfo(
        deviceTypeSet: DeviceTypeSet,
        deviceIndex: Int,
        applianceIndex: Int,
    ): ControllableApplianceInfo {

        val key = ControllableApplianceInfo.createKey(sn, deviceTypeSet, applianceIndex)
        return applianceInfoMap[key] ?: run {
            val applianceInfoNew =
                ControllableApplianceInfo(sn, deviceTypeSet, applianceIndex)
            applianceInfoNew.deviceIndex = deviceIndex
            applianceInfoMap[key] = applianceInfoNew
            applianceInfoNew
        }
    }

    private fun envParaHandle(
        deviceTypeSet: DeviceTypeSet,
        applianceIndexValid: Int,
        paraListValid: List<Int>,
    ) {
        if (deviceTypeSet == DeviceTypeSet.EnvMonitor_W) {
            if (paraListValid.count() >= 8) {
                val airQualityInfo = ControllableApplianceInfo.AirQualityInfo(
                    paraListValid[0],
                    paraListValid[1],
                    paraListValid[2],
                    paraListValid[3],
                    paraListValid[4],
                    paraListValid[5],
                    paraListValid[6],
                    paraListValid[7],
                    sn,
                    deviceTypeSet,
                    applianceIndexValid,
                    0
                )
                this.updatedAirQualityInfoList = listOf(airQualityInfo)
            }
        } else if (deviceTypeSet == DeviceTypeSet.EnvMonitorPlus_W) {
            if (paraListValid.count() >= 12) {
                val airQualityPlusInfo = ControllableApplianceInfo.AirQualityPlusInfo(
                    paraListValid[0],
                    paraListValid[1],
                    paraListValid[2],
                    paraListValid[3],
                    paraListValid[4],
                    paraListValid[5],
                    paraListValid[6],
                    paraListValid[7],
                    paraListValid[8],
                    paraListValid[9],
                    paraListValid[10],
                    paraListValid[11],
                    sn,
                    deviceTypeSet,
                    applianceIndexValid
                )
                this.updatedAirQualityPlusInfoList = listOf(airQualityPlusInfo)
            }
        }
    }

    private fun rbgDeviceParaHandle(paraList: List<Int>): ParaStructure {
        val para = ParaStructure()
        if (paraList.count() >= 4) {
            para.setColor(
                ParaStructure.RgbwType.Red,
                paraList[0]
            )
            para.setColor(
                ParaStructure.RgbwType.Green,
                paraList[1]
            )
            para.setColor(
                ParaStructure.RgbwType.Blue,
                paraList[2]
            )
            para.setColor(
                ParaStructure.RgbwType.Bright,
                paraList[3]
            )
            if (paraList.count() >= 5) {
                para.setColor(
                    ParaStructure.RgbwType.W,
                    paraList[4]
                )
            }
        }
        return para
    }

    private fun irDeviceParaHandle(
        deviceTypeSet: DeviceTypeSet,
        applianceIndex: Int,
        paraList: List<Int>,
    ): List<ControllableApplianceInfo.IrInfo> {
        val irInfoList = mutableListOf<ControllableApplianceInfo.IrInfo>()
        paraList.forEach { para ->
            val irInfo =
                ControllableApplianceInfo.IrInfo(
                    sn, deviceTypeSet, applianceIndex, ParaStructure(
                        para.toULong()
                    )
                )
            irInfoList.add(irInfo)
        }
        return irInfoList.toList()
    }
}

@Serializable
class ControllableApplianceInfo(
    val sn: String,
    var devType: DeviceTypeSet,
    var applianceIndex: Int,
) {
    internal companion object {
        internal fun createKey(sn: String, devType: DeviceTypeSet, applianceIndex: Int): String {
            return sn + devType.raw + applianceIndex
        }
    }

    var deviceIndex: Int? = null
    var fwVersion: FirmwareVersion? = null
    var status: Boolean? = null
    var onlineStatus: Boolean? = null


    var para: ParaStructure? = null

    //设备上的指示灯亮灭设置
    var ledStatus: Boolean? = null

    //上电开关状态
    var powerOnStatus: OnOffStatus? = null

    //调光灯相关参数
    var confInfo: ConfInfo? = null

    var irInfoList: List<IrInfo>? = null

    //环境监测设备相关数据
    var airQualityInfo: AirQualityInfo? = null
    var airQualityPlusInfo: AirQualityPlusInfo? = null

    internal fun createKey(): String {
        return createKey(sn, devType, applianceIndex)
    }

    @Serializable
    class ConfInfo {
        @Serializable
        enum class TurnOnMode(val raw: Int) {
            //开灯时按照 turnOnBrightPercent 设定亮度/色温/颜色
            UserDefault(1),

            //开灯时，按照关灯时的 亮度/色温/颜色来开启
            TurnOffValue(2);

            companion object {
                private val mapping = values().associateBy(TurnOnMode::raw)
                fun fromRaw(raw: Int) = mapping[raw]
            }
        }

        var turnOnMode: TurnOnMode = TurnOnMode.UserDefault
        var turnOnPara: ParaStructure = ParaStructure(0u)
        var minBrightPercent: ParaStructure = ParaStructure(0u) //配置调光设备最低亮度
        var brightAdjSpeed: Int = 0//亮度从0%~100%的总时间，单位为秒（1~15秒），相当于亮度变化速度
        var onOffBrightAdjSpeed: Int = 0//配置开关灯时亮度变化的时间，单位为秒（1~15秒)
        var slowlyOnBrightAdjSpeed: Int = 0 //配置缓开模式下，灯亮度从最低亮度到默认开灯亮度所需的变化时间，单位为分钟（1~60分钟）
        var slowlyOffBrightAdjSpeed: Int = 0//配置缓关模式下，灯亮度从当前亮度到最低亮度所需的变化时间，单位为分钟（1 ~ 60分钟）

        constructor(
            deviceTypeSet: DeviceTypeSet,
            turnOnModeRaw: Int,
            turnOnBrightPercentRaw: Int,
            minBrightPercentRaw: Int,
            brightAdjSpeedRaw: Int,
            onOffBrightAdjSpeedRaw: Int,
            slowlyOnBrightAdjSpeedRaw: Int,
            slowlyOffBrightAdjSpeedRaw: Int,
            turnOnBrightPercentRawList: List<Int>,
        ) {
            TurnOnMode.fromRaw(turnOnModeRaw)?.let { turnOnModeNew ->
                turnOnMode = turnOnModeNew
            }
            if (DeviceTypeClassify.isRGBLampDevice(deviceTypeSet) || DeviceTypeClassify.isRGBWLampDevice(
                    deviceTypeSet
                )
            ) {
                val para = ParaStructure()
                if (turnOnBrightPercentRawList.count() >= 4) {
                    para.setColor(
                        ParaStructure.RgbwType.Red,
                        turnOnBrightPercentRawList[0]
                    )
                    para.setColor(
                        ParaStructure.RgbwType.Green,
                        turnOnBrightPercentRawList[1]
                    )
                    para.setColor(
                        ParaStructure.RgbwType.Blue,
                        turnOnBrightPercentRawList[2]
                    )
                    para.setColor(
                        ParaStructure.RgbwType.Bright,
                        turnOnBrightPercentRawList[3]
                    )
                    if (turnOnBrightPercentRawList.count() >= 5) {
                        para.setColor(
                            ParaStructure.RgbwType.W,
                            turnOnBrightPercentRawList[4]
                        )
                    }
                }
                turnOnPara = para
            } else {
                turnOnPara = ParaStructure(turnOnBrightPercentRaw.toULong())
            }
            minBrightPercent = ParaStructure(minBrightPercentRaw.toULong())
            brightAdjSpeed = when {
                brightAdjSpeedRaw > 15 -> 15
                brightAdjSpeedRaw < 1 -> 1
                else -> brightAdjSpeedRaw
            }
            onOffBrightAdjSpeed = when {
                onOffBrightAdjSpeedRaw > 15 -> 15
                onOffBrightAdjSpeedRaw < 1 -> 1
                else -> onOffBrightAdjSpeedRaw
            }
            slowlyOnBrightAdjSpeed = when {
                slowlyOnBrightAdjSpeedRaw > 60 -> 60
                slowlyOnBrightAdjSpeedRaw < 1 -> 1
                else -> slowlyOnBrightAdjSpeedRaw
            }
            slowlyOffBrightAdjSpeed = when {
                slowlyOffBrightAdjSpeedRaw > 60 -> 60
                slowlyOffBrightAdjSpeedRaw < 1 -> 1
                else -> slowlyOffBrightAdjSpeedRaw
            }
        }
    }

    @Serializable
    class IrInfo(
        val sn: String,
        val deviceTypeSet: DeviceTypeSet,
        val applianceIndex: Int,
        val para: ParaStructure,
    ) {

    }

    @Serializable
    class AirQualityInfo {
        constructor(
            pm1_0Raw: Int,
            pm2_5Raw: Int,
            pm10Raw: Int,
            hchoRaw: Int,
            tvocRaw: Int,
            temperatureRaw: Int,
            humidityRaw: Int,
            aqiRaw: Int,
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            subDevType: Int
        ) {
            val pm1_0: Int = pm1_0Raw
            val pm2_5: Int = pm2_5Raw
            val pm10: Int = pm10Raw
            val hcho: Int = hchoRaw / 1000
            val tvoc: Int = tvocRaw / 1000
            val temperature: Int = temperatureRaw / 10
            val humidity: Int = humidityRaw
            val aqi: Int = aqiRaw
            val sn: String = sn
            val deviceTypeSet: DeviceTypeSet = deviceTypeSet
            val applianceIndex: Int = applianceIndex
            val subDevType: Int = subDevType
        }

    }

    @Serializable
    class AirQualityPlusInfo {
        constructor(
            subDevType: Int,
            pm1_0Raw: Int,
            pm2_5Raw: Int,
            pm10Raw: Int,
            hchoRaw: Int,
            tvocRaw: Int,
            temperatureRaw: Int,
            humidityRaw: Int,
            co2Raw: Int,
            luxRaw: Int,
            aqiRaw: Int,
            battRaw: Int,
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int
        ) {
            val subDevType = subDevType
            val pm1_0: Int = pm1_0Raw
            val pm2_5: Int = pm2_5Raw
            val pm10: Int = pm10Raw
            val hcho: Int = hchoRaw / 1000
            val tvoc: Int = tvocRaw / 1000
            val temperature: Int = temperatureRaw / 10
            val humidity: Int = humidityRaw
            val co2 = co2Raw
            val lux = luxRaw
            val aqi: Int = aqiRaw
            val batt = battRaw
            val sn = sn
            val deviceTypeSet = deviceTypeSet
            val applianceIndex = applianceIndex
        }
    }
}