package com.sentray.kmmprotocolmodule.sdk.parser.parserResult

import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeClassify
import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeSet
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.ProtocolContent
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.Cmd
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.FirmwareVersion
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.ParaStructure
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.TriggerPara
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.UserActionPara
import kotlinx.serialization.Serializable

@Serializable
class DetectableDeviceInfo {
    var installedDetectableApplianceIndexList: List<Int>? = null
    var updatedDetectableApplianceInfoList: List<DetectableApplianceInfo>? = null

    var installedUserActionIndexList: List<Int>? = null

    var updatedDetectableApplianceStatusList: List<DetectableApplianceStatus>? =
        null

    var updatedDetectableApplianceEmergencyStatusList: List<DetectableApplianceEmergencyStatus>? =
        null


    var updatedUserActionList: List<UserAction>? = null

    internal constructor(
        sn: String,
        baseContent: ProtocolContent.BaseContent,
        inputDevContent: ProtocolContent.InputDevContent
    ) {
        inputDevContent.triggerInstalled?.let { triggerInstalled ->
            val installedIndexList = mutableListOf<Int>()
            for (index in 0 until triggerInstalled.count()) {
                if (triggerInstalled[index]) {
                    installedIndexList.add(index)
                }
            }
            installedDetectableApplianceIndexList = installedIndexList.toList()
        }

        inputDevContent.triggerParaList?.let { triggerParaList ->
            baseContent.cmd?.let { cmd ->
                val triggerList: MutableList<DetectableApplianceInfo> = mutableListOf()
                triggerParaList.forEach { triggerPara ->
                    val trigger = DetectableApplianceInfo(
                        sn,
                        cmd,
                        triggerPara,
                        inputDevContent.triggerStatus,
                        inputDevContent.noDisableTrigger,
                        inputDevContent.ver
                    )
                    triggerList.add(trigger)
                }

                updatedDetectableApplianceInfoList = triggerList.toList()
            }
        }

        inputDevContent.actionInstalled?.let { actionInstalled ->
            val installedIndexList = mutableListOf<Int>()
            for (index in 0 until actionInstalled.count()) {
                if (actionInstalled[index]) {
                    installedIndexList.add(index)
                }
            }
            installedUserActionIndexList = installedIndexList.toList()
        }

        inputDevContent.userActionParaList?.let { userActionParaList ->
            val userActionList: MutableList<UserAction> = mutableListOf()
            for (index in 0 until userActionParaList.count()) {
                userActionList.add(UserAction(userActionParaList[index]))
            }
            updatedUserActionList = userActionList.toList()
        }

        if (baseContent.cmd == Cmd.TriggerStatusAck || baseContent.cmd == Cmd.GetAck) {
            inputDevContent.triggerStatus?.let { triggerStatus ->
                val detectableApplianceStatusList: MutableList<DetectableApplianceStatus> =
                    mutableListOf()
                for (index in 0 until triggerStatus.count()) {
                    val detectableApplianceStatus =
                        DetectableApplianceStatus(index, triggerStatus[index])
                    detectableApplianceStatusList.add(detectableApplianceStatus)
                }
                updatedDetectableApplianceStatusList = detectableApplianceStatusList.toList()
            }
        }

        inputDevContent.noDisableTrigger?.let { noDisableTrigger ->
            val detectableApplianceEmergencyStatusList: MutableList<DetectableApplianceEmergencyStatus> =
                mutableListOf()
            for (index in 0 until noDisableTrigger.count()) {
                val status =
                    DetectableApplianceEmergencyStatus(index, noDisableTrigger[index])
                detectableApplianceEmergencyStatusList.add(status)
            }
            updatedDetectableApplianceEmergencyStatusList =
                detectableApplianceEmergencyStatusList.toList()
        }
    }

//    fun freezeSelf() {
//        freeze(this)
//    }
}

@Serializable
class DetectableApplianceInfo {
    internal companion object {
        internal fun createKey(sn: String, devType: DeviceTypeSet, applianceIndex: Int): String {
            return sn + devType.raw + applianceIndex
        }
    }

    var sn: String = ""
    var devType: DeviceTypeSet
    var deviceIndex: Int? = null
    var applianceIndex: Int
    var subIndex: Int? = null
    var bindUserActionIndexList: List<Int>? = null
    var bindDetectableApplianceStatusList: List<DetectableApplianceStatus>? = null
    var activateDelay: Int? = null
    var status: Boolean? = null
    var isEmergency: Boolean? = null
    var fwVersion: FirmwareVersion? = null

    @Deprecated("defaultActionList 已弃用")
    var defaultActionList: List<DefaultAction>? = null

    constructor(
        sn: String,
        devType: DeviceTypeSet,
        applianceIndex: Int
    ) {
        this.sn = sn
        this.devType = devType
        this.applianceIndex = applianceIndex
    }

    internal constructor(
        sn: String,
        cmd: Cmd,
        triggerPara: TriggerPara,
        triggerStatus: List<Boolean>?,
        noDisableTrigger: List<Boolean>?,
        fwVersion: FirmwareVersion?
    ) {
        this.sn = sn
        this.fwVersion = fwVersion
        devType = triggerPara.inputDeviceType
        deviceIndex = triggerPara.inputDeviceIndex
        applianceIndex = triggerPara.triggerIndex
        subIndex = triggerPara.subIndex
        activateDelay = triggerPara.delay
        triggerPara.userActionMask?.let { userActionMask ->
            val bindActionIndexListValid = mutableListOf<Int>()
            for (index in 0 until userActionMask.count()) {
                if (userActionMask[index]) {
                    bindActionIndexListValid.add(index)
                }
            }
            bindUserActionIndexList = bindActionIndexListValid.toList()
        }

        triggerPara.defaultActionMask?.let { defaultActionMask ->
            val defaultActionListValid = mutableListOf<DefaultAction>()
            for (index in 0 until defaultActionMask.count()) {
                if (defaultActionMask[index]) {
                    DefaultAction.fromRaw(index)?.let {
                        defaultActionListValid.add(it)
                    }
                }
            }
            defaultActionList = defaultActionListValid.toList()
        }

        triggerPara.triggerMask?.let { triggerMask ->
            triggerPara.triggerStatus?.let { triggerStatus ->
                val detectableApplianceStatusList: MutableList<DetectableApplianceStatus> =
                    mutableListOf()
                for (index in 0 until triggerMask.count()) {
                    if (triggerMask[index]) {
                        val detectableApplianceStatus = if (index < triggerStatus.count()) {
                            DetectableApplianceStatus(index, triggerStatus[index])
                        } else {
                            DetectableApplianceStatus(index, false)
                        }
                        detectableApplianceStatusList.add(detectableApplianceStatus)
                    }
                }

                bindDetectableApplianceStatusList = detectableApplianceStatusList.toList()
            }
        }

        if (cmd == Cmd.GetTriggerAck) {
            applianceIndex?.let { applianceIndex ->
                noDisableTrigger?.let { noDisableTrigger ->
                    if (applianceIndex < noDisableTrigger.count()) {
                        isEmergency = noDisableTrigger[applianceIndex]
                    }
                }
                triggerStatus?.let { triggerStatus ->
                    if (applianceIndex < triggerStatus.count()) {
                        status = triggerStatus[applianceIndex]
                    }
                }
            }
        }
    }

    internal fun createKey(): String {
        return createKey(sn, devType, applianceIndex)
    }
}

@Serializable
class UserAction {
    var actionIndex: Int = 0
    var devType: DeviceTypeSet = DeviceTypeSet.Unknown
    var applianceParaList: List<UserActionAppliancePara>? = null

    internal constructor(userActionPara: UserActionPara) {
        actionIndex = userActionPara.actionIndex
        devType = userActionPara.devType

        val rlyMask = userActionPara.rlyMask ?: listOf()
        val rlyAction = userActionPara.rlyAction ?: listOf()
        val paraMask = userActionPara.paraMask ?: listOf()
        val paraList = userActionPara.paraList ?: listOf()

        if (rlyMask.isNotEmpty() && paraMask.isNotEmpty()) {
            var paraListReadIndex = 0
            val applianceParaListNew: MutableList<UserActionAppliancePara> =
                mutableListOf()
            for (index in 0 until rlyMask.count()) {
                if (rlyMask[index] || (index < paraMask.count() && paraMask[index])) {
                    val userActionAppliancePara = UserActionAppliancePara()
                    userActionAppliancePara.applianceIndex = index

                    if (rlyMask[index]) {
                        userActionAppliancePara.action = when (index < rlyAction.count()) {
                            true -> rlyAction[index]
                            false -> false
                        }
                    }
                    if (index < paraMask.count() && paraMask[index]) {
                        //若为 RGBW 设备，则 paraList 仅供一个设备使用
                        if (DeviceTypeClassify.isRGBLampDevice(devType) || DeviceTypeClassify.isRGBWLampDevice(
                                devType
                            )
                        ) {

                            if (paraList.count() >= 4) {
                                userActionAppliancePara.para.setColor(
                                    ParaStructure.RgbwType.Red,
                                    paraList[0].getRawValue().toInt()
                                )
                                userActionAppliancePara.para.setColor(
                                    ParaStructure.RgbwType.Green,
                                    paraList[1].getRawValue().toInt()
                                )
                                userActionAppliancePara.para.setColor(
                                    ParaStructure.RgbwType.Blue,
                                    paraList[2].getRawValue().toInt()
                                )
                                userActionAppliancePara.para.setColor(
                                    ParaStructure.RgbwType.Bright,
                                    paraList[3].getRawValue().toInt()
                                )
                                if (paraList.count() >= 5) {
                                    userActionAppliancePara.para.setColor(
                                        ParaStructure.RgbwType.W,
                                        paraList[4].getRawValue().toInt()
                                    )
                                }
                            }
                            applianceParaListNew.add(userActionAppliancePara)
                            break
                        } else {
                            userActionAppliancePara.para =
                                ParaStructure(paraList[paraListReadIndex].getRawValue())
                            paraListReadIndex++
                        }
                    } else {
                        userActionAppliancePara.para = ParaStructure(0u)
                    }

                    applianceParaListNew.add(userActionAppliancePara)
                }
            }

            applianceParaList = applianceParaListNew.toList()
        }
    }
}

@Serializable
data class DetectableApplianceStatus(
    val applianceIndex: Int,
    val status: Boolean,
)

@Serializable
data class DetectableApplianceEmergencyStatus(
    val applianceIndex: Int,
    val emergencyStatus: Boolean,
)

@Serializable
class UserActionAppliancePara {
    var applianceIndex: Int = 0
    var action: Boolean = false
    var para = ParaStructure(0u)
}

enum class DefaultAction(val raw: Int) {
    @Deprecated("暂未实现")
    Alarm(1),

    @Deprecated("暂未实现")
    PhoneMessage(3),

    @Deprecated("暂未实现")
    PhoneCall(4),
    AppPushNotification(2);


    companion object {
        private val mapping = values().associateBy(DefaultAction::raw)
        fun fromRaw(raw: Int) = mapping[raw]
    }
}