package org.brewchain.core.crypto.cwv;

import org.brewchain.core.crypto.cwv.jce.SpongyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongycastle.crypto.Digest;
import org.spongycastle.crypto.digests.RIPEMD160Digest;
import org.spongycastle.jcajce.provider.digest.SHA3;
import org.spongycastle.pqc.math.linearalgebra.ByteUtils;
import org.spongycastle.util.encoders.Hex;

import java.awt.*;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.Security;
import java.util.Random;

import static java.util.Arrays.copyOfRange;

public class HashUtil {
    private static final Logger LOG = LoggerFactory.getLogger(HashUtil.class);

    private static final Provider CRYPTO_PROVIDER;

    private static final String HASH_256_ALGORITHM_NAME;
    private static final String HASH_512_ALGORITHM_NAME;

    private static final MessageDigest sha256digest;

    static {
        Security.addProvider(SpongyCastleProvider.getInstance());
        CRYPTO_PROVIDER = Security.getProvider("SC");
//        CRYPTO_PROVIDER =SpongyCastleProvider.getInstance();
        HASH_256_ALGORITHM_NAME = "ETH-KECCAK-256";
        HASH_512_ALGORITHM_NAME = "ETH-KECCAK-512";
        try {
            sha256digest = SHA3.Digest256.getInstance("SHA-256");//MessageDigest.getInstance("SHA-256");
//		} catch (NoSuchAlgorithmException e) {
        } catch (Exception e) {
            LOG.error("Can't initialize HashUtils", e);
            throw new RuntimeException(e); // Can't happen.
        }
    }

    /**
     * @param input
     *            - data for hashing
     * @return - sha256 hash of the data
     */
    public static byte[] sha256(byte[] input) {
        try {

            MessageDigest dig = SHA3.Digest256.getInstance("SHA-256");//MessageDigest.getInstance("SHA-256");
            return dig.digest(input);
//		} catch (NoSuchAlgorithmException e) {
        } catch (Exception e) {
            LOG.error("Can't initialize HashUtils", e);
            throw new RuntimeException(e); // Can't happen.
        }
    }

    public static byte[] sha3(byte[] input) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance(HASH_256_ALGORITHM_NAME, CRYPTO_PROVIDER);
            digest.update(input);
            return digest.digest();
        } catch (NoSuchAlgorithmException e) {
            LOG.error("Can't find such algorithm", e);
            throw new RuntimeException(e);
        }

    }

    public static byte[] sha3(byte[] input1, byte[] input2) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance(HASH_256_ALGORITHM_NAME, CRYPTO_PROVIDER);
            digest.update(input1, 0, input1.length);
            digest.update(input2, 0, input2.length);
            return digest.digest();
        } catch (NoSuchAlgorithmException e) {
            LOG.error("Can't find such algorithm", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * hashing chunk of the data
     *
     * @param input
     *            - data for hash
     * @param start
     *            - start of hashing chunk
     * @param length
     *            - length of hashing chunk
     * @return - keccak hash of the chunk
     */
    public static byte[] sha3(byte[] input, int start, int length) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance(HASH_256_ALGORITHM_NAME, CRYPTO_PROVIDER);
            digest.update(input, start, length);
            return digest.digest();
        } catch (NoSuchAlgorithmException e) {
            LOG.error("Can't find such algorithm", e);
            throw new RuntimeException(e);
        }
    }

    public static byte[] sha512(byte[] input) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance(HASH_512_ALGORITHM_NAME, CRYPTO_PROVIDER);
            digest.update(input);
            return digest.digest();
        } catch (NoSuchAlgorithmException e) {
            LOG.error("Can't find such algorithm", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * @param data
     *            - message to hash
     * @return - reipmd160 hash of the message
     */
    public static byte[] ripemd160(byte[] data) {
        Digest digest = new RIPEMD160Digest();
        if (data != null) {
            byte[] resBuf = new byte[digest.getDigestSize()];
            digest.update(data, 0, data.length);
            digest.doFinal(resBuf, 0);
            return resBuf;
        }
        throw new NullPointerException("Can't hash a NULL value");
    }

    /**
     * Calculates RIGTMOST160(SHA3(input)). This is used in address
     * calculations. *
     *
     * @param input
     *            - data
     * @return - 20 right bytes of the hash keccak of the data
     */
    public static byte[] sha3omit12(byte[] input) {
        byte[] hash = sha3(input);
        return copyOfRange(hash, 12, hash.length);
    }

    /**
     * @see #doubleDigest(byte[], int, int)
     *
     * @param input
     *            -
     * @return -
     */
    public static byte[] doubleDigest(byte[] input) {
        return doubleDigest(input, 0, input.length);
    }

    /**
     * Calculates the SHA-256 hash of the given byte range, and then hashes the
     * resulting hash again. This is standard procedure in Bitcoin. The
     * resulting hash is in big endian form.
     *
     * @param input
     *            -
     * @param offset
     *            -
     * @param length
     *            -
     * @return -
     */
    public static byte[] doubleDigest(byte[] input, int offset, int length) {
        synchronized (sha256digest) {
            sha256digest.reset();
            sha256digest.update(input, offset, length);
            byte[] first = sha256digest.digest();
            return sha256digest.digest(first);
        }
    }

    /**
     * @return - generate random 32 byte hash
     */
    public static byte[] randomHash() {

        byte[] randomHash = new byte[32];
        Random random = new Random();
        random.nextBytes(randomHash);
        return randomHash;
    }

    public static String shortHash(byte[] hash) {
        return Hex.toHexString(hash).substring(0, 6);
    }

    public static void main(String[] args) {
//        //sha256( sha256(index + "address") + address )
//        //sha256( sha256(1 + "0") + "0" )
//        String hash = new String(Hex.encode(HashUtil.sha256(ByteUtils.concatenate(
//                HashUtil.sha256(ByteUtils.concatenate(
//                        BigInteger.valueOf(1).toByteArray(),
//                        BigInteger.valueOf(0).toByteArray())),
//                BigInteger.valueOf(0).toByteArray()))));
//
//        System.out.println("hash="+hash);
//       // keccak256(uint256(0) . keccak256(uint256(0) . uint256(1))) + 1:
//        String hash2= new String(Hex.encode(HashUtil.sha3(ByteUtils.concatenate(
//                BigInteger.valueOf(0).toByteArray(),
//                HashUtil.sha3(
//                        ByteUtils.concatenate(BigInteger.valueOf(0).toByteArray(),
//                        BigInteger.valueOf(1).toByteArray())
//                )
//        ))));
//        System.out.println("hash2="+hash2);
//        String hash2= new String(Hex.encode(HashUtil.sha3(
//                Hex.decode("0000000000000000000000000000000000000000000000000000000000000000"+
//                        new String(Hex.encode(HashUtil.sha3(Hex.decode(
//                                "0000000000000000000000000000000000000000000000000000000000000000"+
//                                        "0000000000000000000000000000000000000000000000000000000000000001")
//                        ))
//                        )
//                ))));
//
//        System.out.println("hash2="+hash2);


    }
}
