package io.github.devsecops.engine.domain.docker.commands;

import io.github.devsecops.engine.core.InstructionInvoker;
import io.github.devsecops.engine.core.contract.Command;
import io.github.devsecops.engine.core.contract.Executor;
import io.github.devsecops.engine.core.contract.Instruction;
import io.github.devsecops.engine.core.model.Environment;
import io.github.devsecops.engine.domain.artifact.model.ArtifactRepositoryType;
import io.github.devsecops.engine.domain.artifact.model.NexusVariable;
import io.github.devsecops.engine.domain.docker.instructions.DockerBuildImageInstruction;
import io.github.devsecops.engine.domain.docker.instructions.DockerLoginInstruction;
import io.github.devsecops.engine.domain.docker.instructions.DockerPushImageInstruction;
import io.github.devsecops.engine.domain.docker.model.DockerVariables;
import io.github.devsecops.engine.domain.pom.model.Pom;
import io.github.devsecops.engine.domain.pom.utils.JarFullPathRetriever;
import io.github.devsecops.engine.domain.resolver.strategy.Resolver;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.List;

@Component
public class BuildImageCommand implements Command {



    @Autowired
    private Executor executor;
    @Setter
    private ArtifactRepositoryType repositoryType;
    @Setter
    private Environment env;
    @Setter
    private Resolver resolver;

    @Override
    public void execute() throws Exception {
        DockerBuildImageInstruction.DockerBuildImageInstructionBuilder imageBuilder = DockerBuildImageInstruction.builder()
                .imageUrl(resolver.resolve(DockerVariables.DOCKER_IMAGE_FULL_URL.getName()))
                .passphrase(resolver.resolve(DockerVariables.DOCKER_GPG_PASSPHRASE.getName()))
                .port(resolver.resolve(DockerVariables.DOCKER_SERVICE_PORT.getName()));

        DockerPushImageInstruction.DockerPushImageInstructionBuilder pushBuilder = DockerPushImageInstruction.builder()
                .imageUrl(resolver.resolve(DockerVariables.DOCKER_IMAGE_FULL_URL.getName()));

        String type = "";
        if (repositoryType != null) {
            if (ArtifactRepositoryType.RELEASE.equals(repositoryType)) {
                imageBuilder.artifactUrl(getReleaseArtifactUrl());
            } else {
                imageBuilder.artifactUrl(getSnapshotArtifactUrl());
            }
            type = new String(":" + env + "-" + repositoryType).toLowerCase();
        }
        imageBuilder.type(type);
        pushBuilder.type(type);
        Instruction login = DockerLoginInstruction.builder().password(resolver.resolve(DockerVariables.DOCKER_HUB_PASSWORD.getName())).build();
        InstructionInvoker.init(executor).append(login).append(imageBuilder.build()).append(pushBuilder.build()).execute();
    }

    private String getReleaseArtifactUrl() {
        Pom pom = Pom.getINSTANCE();
        final String nexusUrl = resolver.resolve(NexusVariable.NEXUS_RELEASE_REPOSITORY_URL.getName());
        final String pomVersion = pom.getVersion().replace("-SNAPSHOT", "");
        final String path = pom.getGroupId().replace(".", "/");
        return String.format("%s/%s/%s/%s/%s-%s.jar", nexusUrl, path, pom.getArtifactId(), pom, pom.getArtifactId(), pomVersion);
    }

    private String getSnapshotArtifactUrl() {
        Pom pom = Pom.getINSTANCE();
        final String nexusUrl = resolver.resolve(NexusVariable.NEXUS_SNAPSHOT_REPOSITORY_URL.getName());
        final String pomVersion = pom.getVersion();
        final String path = pom.getGroupId().replace(".", "/");
        final String nexusRepositoryUrl = String.format("%s/%s/%s/%s", nexusUrl, path, pom.getArtifactId(), pomVersion);
        return JarFullPathRetriever.retrieve(nexusRepositoryUrl);
    }


    @Override
    public void rollback() throws Exception {
        throw new UnsupportedOperationException("No rollback is specified for artifact commands");
    }
}
