package io.github.kaststream.api.v2;

import io.github.kaststream.api.v2.facade.KastRow;
import java.io.Serializable;
import java.util.Map;
import org.apache.avro.Schema;

public interface Filter extends Serializable {
  /**
   * The filter function that evaluates the predicate.
   *
   * <p><strong>IMPORTANT:</strong> The system assumes that the function does not modify the
   * elements on which the predicate is applied. Violating this assumption can lead to incorrect
   * results.
   *
   * @param kRow The value to be filtered.
   * @return True for values that should be retained, false for values to be filtered out.
   * @throws Exception This method may throw exceptions. Throwing an exception will cause the
   *     operation to fail and may trigger recovery.
   */
  public boolean filter(final KastRow kRow) throws Exception;

  /**
   * Declare the data type information of the data produced by your function.
   *
   * @param schema the input row schema
   * @return the produced data typeInfo
   */
  public Schema getProducedType(final Schema schema);

  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @param schema the input row schema
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(final Map<String, String> configuration, final Schema schema)
      throws Exception {}
  ;
  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @param schema the input row schema
   * @param meta the udf configuration parameters
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(
      final Map<String, String> configuration, final Schema schema, Map<String, Object> meta)
      throws Exception {
    throw new UnsupportedOperationException("open with meta not implemented");
  }
  ;
  /**
   * Tear-down method for the user code. It is called after the last call to the main working
   * methods. For functions that are part of an iteration, this method will be invoked after each
   * iteration superstep.
   *
   * <p>This method can be used for clean up work.
   *
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void close() throws Exception {}
  ;
}
