package io.github.kaststream.api.v2;

import io.github.kaststream.api.v2.facade.*;
import java.io.Serializable;
import java.util.Map;
import org.apache.avro.Schema;

/** The process function (engine low level operator) */
public interface Processor extends Serializable {

  /**
   * @param kRow in value
   * @param context the ProcessFunction.Context to eventually emit side-output streams
   * @param collector the out row collector
   * @throws Exception in case of problem. This will stop your stream.
   */
  public void processElement(
      final KastRow kRow,
      final KastProcessorContext context,
      final KastCollector collector,
      KastRowBuilder kastRowBuilder)
      throws Exception;

  /**
   * Declare the data type information of the data produced by your function.
   *
   * @param schema the input row schema
   * @return the output row type information
   */
  public Schema getProducedType(final Schema schema);

  /**
   * Declare the data type information of the sides out data produced by your function.
   *
   * @return the sides output type information
   */
  public default Map<String, Schema> getSidesOutProducedType(final Schema schema) {
    return null;
  }

  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @param schema the input row schema
   * @param runtimeContext Context that contains information about the UDF's runtime
   * @param sidesOutputTag Side output typeinfo and tag dict if any
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(
      final Map<String, String> configuration,
      final Schema schema,
      final KastRuntimeContext runtimeContext,
      final Map<String, KastOutputTag> sidesOutputTag)
      throws Exception {}
  ;
  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @param schema the input row schema
   * @param runtimeContext Context that contains information about the UDF's runtime
   * @param sidesOutputTag Side output typeinfo and tag dict if any
   * @param meta function meta
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(
      final Map<String, String> configuration,
      final Schema schema,
      final KastRuntimeContext runtimeContext,
      final Map<String, KastOutputTag> sidesOutputTag,
      final Map<String, Object> meta)
      throws Exception {
    throw new UnsupportedOperationException("open with meta not implemented");
  }
  ;
  /**
   * Tear-down method for the user code. It is called after the last call to the main working
   * methods. For functions that are part of an iteration, this method will be invoked after each
   * iteration superstep.
   *
   * <p>This method can be used for clean up work.
   *
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void close() throws Exception {};
}
