/*
 * Decompiled with CFR 0.152.
 */
package com.monmonkeygroup.openapi.client;

import com.monmonkeygroup.openapi.OpenApiException;
import com.monmonkeygroup.openapi.client.IClientConn;
import com.monmonkeygroup.openapi.client.ISocketServiceListener;
import com.monmonkeygroup.openapi.client.Profile;
import com.monmonkeygroup.openapi.client.PushHandler;
import com.monmonkeygroup.openapi.client.WebsocketConn;
import com.monmonkeygroup.openapi.protocol.Message;
import com.monmonkeygroup.openapi.protocol.Packet;
import com.monmonkeygroup.openapi.protocol.v1.ProtocolV1;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Client
implements ISocketServiceListener,
AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(Client.class);
    private final Profile profile;
    private final URI uri;
    private final IClientConn conn;
    private final AtomicBoolean running = new AtomicBoolean(true);
    private final AtomicBoolean autoReconnect = new AtomicBoolean(true);
    private final AtomicLong nextPacketTxId = new AtomicLong(1L);
    private final Map<Long, CompletableFuture<Packet>> pendingRequests = new ConcurrentHashMap<Long, CompletableFuture<Packet>>();
    private final Map<String, Set<PushHandler>> pushHandlers = new ConcurrentHashMap<String, Set<PushHandler>>();
    private Timer timer;
    private Timer heartbeatTimer;
    private long heartbeatInterval = 10000L;
    private long retryInterval = 1000L;
    private int alreadyRetry = 0;
    private int maxRetry = 0;
    private AfterConnected afterConnected;
    private boolean firstConnect = true;
    private int requestTimeout = 15000;

    public Client(Profile profile) {
        try {
            this.uri = new URI(profile.getUri());
        }
        catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
        this.profile = profile;
        this.conn = new WebsocketConn(this.uri, new ProtocolV1());
        this.conn.addListener(this);
    }

    public void init() throws OpenApiException {
        this.conn.init();
        this.auth();
        this.heartbeatTimer = new Timer();
        this.heartbeatTimer.scheduleAtFixedRate(new TimerTask(){

            @Override
            public void run() {
                Client.this.heartbeat();
            }
        }, this.heartbeatInterval, this.heartbeatInterval);
    }

    public <T> T doRequest(String actionName, Object request, Class<T> replyClazz) throws OpenApiException {
        if (!this.conn.isOpen()) {
            throw new OpenApiException(0, "Request rejected: Client not connected");
        }
        Packet requestPacket = Packet.CreateRequest(actionName, this.nextPacketTxId.getAndIncrement(), request);
        Packet replyPacket = this.syncDo(requestPacket);
        if (replyPacket.getErrCode() != 0) {
            throw new OpenApiException(replyPacket.getErrCode(), "");
        }
        return replyPacket.deserialize(replyClazz);
    }

    private Packet syncDo(Packet requestPacket) throws OpenApiException {
        Packet replyPacket;
        try {
            this.sendPacket(requestPacket);
            CompletableFuture future = new CompletableFuture();
            this.pendingRequests.put(requestPacket.getTxId(), future);
            replyPacket = (Packet)future.get(this.requestTimeout, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            throw new OpenApiException(0, "Reply timeout");
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof OpenApiException) {
                throw (OpenApiException)e.getCause();
            }
            log.error(e.getMessage(), (Throwable)e);
            throw new OpenApiException(0, e.toString());
        }
        catch (Exception e) {
            log.error(e.getMessage(), (Throwable)e);
            throw new OpenApiException(0, e.toString());
        }
        finally {
            this.pendingRequests.remove(requestPacket.getTxId());
        }
        return replyPacket;
    }

    public void sendPacket(Packet packet) {
        this.conn.sendPacket(packet);
    }

    public void addPushHandler(String actionName, PushHandler handler) {
        Set set = this.pushHandlers.computeIfAbsent(actionName, k -> ConcurrentHashMap.newKeySet());
        set.add(handler);
    }

    private void heartbeat() {
        if (!this.conn.isOpen()) {
            return;
        }
        Message.HeartbeatRequest request = new Message.HeartbeatRequest(System.currentTimeMillis());
        try {
            this.doRequest("heartbeatRequest", request, Message.HeartbeatReply.class);
        }
        catch (OpenApiException openApiException) {
            // empty catch block
        }
    }

    private void auth() throws OpenApiException {
        Message.LoginRequest request = new Message.LoginRequest(this.profile.getUser(), this.profile.getToken());
        this.doRequest("login", request, Message.LoginReply.class);
    }

    @Override
    public void onConnected() {
        if (this.firstConnect) {
            this.firstConnect = false;
            return;
        }
        new Thread(() -> {
            try {
                this.auth();
                if (null != this.afterConnected) {
                    this.afterConnected.onAfterConnected(this);
                }
            }
            catch (OpenApiException e) {
                log.error(e.getMessage());
                this.conn.close("Login failed");
            }
        }).start();
    }

    @Override
    public void onMessage(Packet packet) {
        if (Objects.equals(packet.getActionType(), "update")) {
            this.handlePush(packet);
        } else if (Objects.equals(packet.getActionType(), "reply")) {
            this.handleReply(packet);
        }
    }

    private void handlePush(Packet packet) {
        Set<PushHandler> handlers = this.pushHandlers.get(packet.getActionName());
        if (null == handlers) {
            return;
        }
        handlers.forEach(handler -> handler.handlePush(packet));
    }

    private void handleReply(Packet packet) {
        CompletableFuture<Packet> future = this.pendingRequests.get(packet.getTxId());
        if (null == future) {
            log.warn("no reply handler for txId: {}", (Object)packet.getTxId());
            return;
        }
        if (future.isCancelled() || future.isDone()) {
            log.warn("duplicate reply of txId: {}", (Object)packet.getTxId());
            return;
        }
        future.complete(packet);
    }

    @Override
    public void onClose(String reason) {
        for (CompletableFuture<Packet> future : this.pendingRequests.values()) {
            future.completeExceptionally(new OpenApiException(0, "Request failed due to connection loss"));
        }
        if (!this.running.get() || !this.autoReconnect.get()) {
            return;
        }
        log.debug("reconnect for conn closed: {}", (Object)reason);
        this.startTimer();
    }

    private void reconnect() {
        try {
            this.conn.reconnectSocket();
            this.alreadyRetry = 0;
            this.stopTimer();
            return;
        }
        catch (Exception exception) {
            ++this.alreadyRetry;
            if (this.maxRetry != 0 && this.alreadyRetry >= this.maxRetry) {
                this.stopTimer();
            }
            return;
        }
    }

    public synchronized void startTimer() {
        if (null != this.timer) {
            return;
        }
        this.timer = new Timer("SocketReconnectingThread-" + System.currentTimeMillis());
        this.timer.scheduleAtFixedRate(new TimerTask(){

            @Override
            public void run() {
                Client.this.reconnect();
            }
        }, 0L, this.retryInterval);
    }

    public synchronized void stopTimer() {
        this.timer.cancel();
        this.timer = null;
    }

    @Override
    public synchronized void close() {
        if (!this.running.get()) {
            return;
        }
        this.running.set(false);
        this.autoReconnect.set(false);
        this.heartbeatTimer.cancel();
        if (null != this.conn) {
            this.conn.close("close by client");
        }
        if (null != this.timer) {
            this.stopTimer();
        }
    }

    public void setHeartbeatInterval(long heartbeatInterval) {
        this.heartbeatInterval = heartbeatInterval;
    }

    public void setRetryInterval(long retryInterval) {
        this.retryInterval = retryInterval;
    }

    public void setMaxRetry(int maxRetry) {
        this.maxRetry = maxRetry;
    }

    public void setRequestTimeout(int requestTimeout) {
        this.requestTimeout = requestTimeout;
    }

    public void setAfterConnected(AfterConnected afterConnected) {
        this.afterConnected = afterConnected;
    }

    public static interface AfterConnected {
        public void onAfterConnected(Client var1);
    }
}

