package pluginloader.api

import java.lang.reflect.Field
import java.lang.reflect.Method
import kotlin.reflect.KClass

interface LoaderPlugin {
    /**
     * Return plugin name, like jar name
     * 'example.jar' -> name: 'example'
     */
    val name: String

    @Deprecated("Use another methods", level = DeprecationLevel.ERROR)
    fun load(kClass: KClass<*>)

    /**
     * Calling [handler] once on unload plugin with /plu u example or on stop server
     */
    fun unloadHandler(handler: () -> Unit)

    /**
     * Delete [handler] from [unloadHandler]
     */
    fun removeUnloadHandler(handler: () -> Unit)

    /**
     * Run [task] in main thread
     */
    fun task(task: () -> Unit)

    /**
     * Allow DI in plugins
     * ```
     * @Retention(AnnotationRetention.RUNTIME)
     * annotation class InjectAPI
     *
     * class Inject(val count: Int = 0)
     *
     * @Load internal fun load(plugin: Plugin){
     *  plugin.fieldReplacer(InjectAPI::class){plu, annotation, input ->
     *      Inject((input?.count ?: 0) + 1)
     *  }
     * }
     * ```
     * Use DI
     * ```
     * @InjectAPI internal var inject = Inject(2)
     *
     * @Load
     * internal fun load(){
     *  println("Inject: ${inject.count}")//'Inject: 3'
     * }
     * ```
     * @param kClass Annotation class
     * @param handler Called on load plugin field with annotation [kClass]
     */
    fun <T: Annotation, Obj> fieldReplacer(kClass: KClass<T>, handler: (plugin: LoaderPlugin, annotation: T, input: Obj) -> Obj)

    @Deprecated("Use fieldReplacer")
    fun <T: Annotation> fieldHandler(kClass: KClass<T>, handler: (Field, T, LoaderPlugin) -> Unit)

    /**
     * Like [fieldReplacer] but on methods
     */
    fun <T: Annotation> methodHandler(kClass: KClass<T>, handler: (Method, T, LoaderPlugin) -> Unit)

    /**
     * Register crossplatform command. Can be used only with privileged access
     * See also [Cmd]
     * ```
     * plugin.cmd("command", {sender, args -> sender.sendMessage("hi"})
     * ```
     * @param sender [CmdSender]
     * @param args [Args] /example a b -> arrayOf("a", "b")
     */
    fun cmd(command: String, cmd: (CmdSender, Args) -> Unit, vararg aliases: String)
}

fun <T: Annotation, Obj> LoaderPlugin.fieldReplacer(kClass: KClass<T>, obj: Obj){
    fieldReplacer<T, Obj>(kClass){_, _, _ -> obj}
}

inline fun <T: Annotation, Obj> LoaderPlugin.fieldReplacer(kClass: KClass<T>, crossinline handler: (plugin: LoaderPlugin) -> Obj){
    fieldReplacer<T, Obj>(kClass){plugin, _, _ -> handler(plugin)}
}

fun <T> LoaderPlugin.loaded(list: MutableList<T>, value: T){
    list.add(value)
    unloadHandler{list.remove(value)}
}

fun <K, V> LoaderPlugin.loaded(map: MutableMap<K, V>, key: K, value: V){
    map[key] = value
    unloadHandler{map.remove(key)}
}

fun LoaderPlugin.throwReadable(message: String, source: Throwable? = null): Nothing{
    throw ReadableException(source, message, this)
}