package io.github.qsy7.linux.builder.impl.service.module;

import io.github.qsy7.download.api.model.Download;
import io.github.qsy7.download.api.service.DownloadService;
import io.github.qsy7.file.api.service.DirectoryCopierService;
import io.github.qsy7.linux.builder.api.model.configuration.BuildConfiguration;
import io.github.qsy7.linux.builder.api.model.configuration.ISOConfiguration;
import io.github.qsy7.linux.builder.api.service.BuildService;
import io.github.qsy7.linux.builder.impl.service.annotation.ModuleSupports;
import io.github.qsy7.linux.builder.impl.service.enumeration.DistributionConfiguration;
import io.github.qsy7.linux.builder.impl.service.util.configuration.YamlConfigurer;
import io.github.qsy7.shell.api.enumeration.MountAction;
import io.github.qsy7.shell.api.enumeration.VFSType;
import io.github.qsy7.shell.api.model.MountCommand;
import io.github.qsy7.shell.api.model.MountPoint;
import io.github.qsy7.shell.api.service.MountService;
import io.github.qsy7.shell.api.service.ShellExecutionService;
import java.io.File;
import javax.inject.Inject;

/**
 * This module works for any distribution that provides an ISO image. 1. download ISO image 2.
 * extract root FS from ISO image
 */
@ModuleSupports(
    distribution = DistributionConfiguration.Linux,
    configurer = YamlConfigurer.class,
    configurationClass = ISOConfiguration.class)
public class IsoModule extends AbstractSingleModule<ISOConfiguration> {
  public static final String ISO_FILE = "ISO-FILE";
  public static final String ISO_ROOT_CONTENTS = "ISO-ROOT-CONTENTS";

  protected final DownloadService downloadService;
  protected final MountService mountService;
  protected final DirectoryCopierService directoryCopierService;

  @Inject
  public IsoModule(
      BuildService buildService,
      BuildConfiguration buildConfiguration,
      DistributionConfiguration distributionConfiguration,
      DownloadService downloadService,
      MountService mountService,
      DirectoryCopierService directoryCopierService,
      ShellExecutionService shellExecutionService) {
    super(buildService, buildConfiguration, distributionConfiguration);
    this.downloadService = downloadService;
    this.mountService = mountService;
    this.directoryCopierService = directoryCopierService;
    this.shellExecutionService = shellExecutionService;
  }

  @Override
  public void document() {
    // super.document();
    // tex.write_table(documentation_directory, 'Stage3Module AbstractConfiguration', ['Key',
    // 'Value'], [.2, .8],
    //                [['Stage3Module URI', self.stage3_uri], ['Stage3Module Checksum URI',
    // self.stage3_checksum_uri], ['Stage3Module Checksum', self.stage3_checksum]])
  }

  @Override
  public void doRun() throws Exception {
    setupRoot();
  }

  protected boolean setupRoot() throws Exception {
    // no need to re-runInChroot stage3 setup, instead, runInChroot the other configuration
    // TODO: move that into its own module?
    final File bash = new File(buildConfiguration.getRootDirectory() + "/bin/bash");
    if (bash.exists()) {
      return false;
    }

    extractRootFS(downloadISO());
    return true;
  }

  protected File downloadISO() throws Exception {
    return downloadService.download(
        new Download(configuration.getDownloadURI(), configuration.getDownloadChecksum()));
  }

  protected final ShellExecutionService shellExecutionService;

  protected void extractRootFS(final File isoFile) throws Exception {
    prepareDirectory(new File(buildConfiguration.getRootDirectory()));

    final MountPoint isoFileMountPoint = getImageMountPoint(isoFile);
    final MountPoint targetFileMountPoint = getImageContentsMountPoint(isoFileMountPoint);
    try {
      prepareDirectory(
          new File(
              buildConfiguration.getBuildDirectory() + "/" + isoFileMountPoint.getMountPoint()));

      final File targetImageContents =
          new File(
              buildConfiguration.getBuildDirectory() + "/" + targetFileMountPoint.getMountPoint());
      prepareDirectory(targetImageContents);

      doMount(isoFileMountPoint);
      doMount(targetFileMountPoint);

      directoryCopierService.copy(
          targetImageContents.toPath(), new File(buildConfiguration.getRootDirectory()).toPath());
    } finally {
      doUnmount(targetFileMountPoint);
      doUnmount(isoFileMountPoint);
    }
  }

  protected void prepareDirectory(final File directory) {
    if (!directory.exists()) {
      directory.mkdirs();
    }
  }

  // ensure that we don't collide with other build processes
  protected MountPoint getImageMountPoint(final File isoFile) {
    return new MountPoint(
        "/images/"
            + buildConfiguration.getScmConfiguration().getTag().getTag()
            + "/"
            + buildConfiguration.getVariant(),
        isoFile.getAbsolutePath(),
        VFSType.AUTO);
  }

  protected MountPoint getImageContentsMountPoint(final MountPoint isoFileMountPoint) {
    return new MountPoint(
        "/image-contents/"
            + buildConfiguration.getScmConfiguration().getTag().getTag()
            + "/"
            + buildConfiguration.getVariant(),
        buildConfiguration.getBuildDirectory()
            + File.separator
            + isoFileMountPoint.getMountPoint()
            + configuration.getRootFSPath(),
        VFSType.AUTO);
  }

  protected void doMount(final MountPoint mountPoint) throws Exception {
    mountService.execute(
        new MountCommand()
            .withMountAction(MountAction.Mount)
            .withMountPoint(mountPoint)
            .withRootPath(buildConfiguration.getBuildDirectory()));
  }

  protected void doUnmount(final MountPoint mountPoint) throws Exception {
    mountService.execute(
        new MountCommand()
            .withMountAction(MountAction.Unmount)
            .withMountPoint(mountPoint)
            .withRootPath(buildConfiguration.getBuildDirectory()));
  }
}
