package io.github.qsy7.ssh.executor;

import io.github.qsy7.property.api.annotation.Property;
import io.github.qsy7.ssh.AbstractSSHService;
import io.github.qsy7.ssh.SSHCommandProcessExecution;
import io.github.qsy7.ssh.api.SSHPublicKeyPath;
import io.github.qsy7.ssh.api.model.command.SSHCommand;
import io.github.qsy7.ssh.property.InterruptGracePeriodUnits;
import io.github.qsy7.ssh.property.InterruptGracePeriodValue;
import java.io.IOException;
import java.time.temporal.ChronoUnit;
import net.schmizz.sshj.SSHClient;
import net.schmizz.sshj.connection.channel.direct.Session;

public class DefaultSSHCommandExecutor extends AbstractSSHService<SSHCommand> {
  // this doesn't work on guice because Guice requires enum constants to follow Case Caps
  protected final ChronoUnit interruptGracePeriodUnits;
  protected final long interruptGracePeriodValue;

  public DefaultSSHCommandExecutor(
      @Property(SSHPublicKeyPath.class) final String sshPublicKeyPath,
      SSHCommand command,
      @Property(InterruptGracePeriodUnits.class) ChronoUnit interruptGracePeriodUnits,
      @Property(InterruptGracePeriodValue.class) long interruptGracePeriodValue) {
    super(sshPublicKeyPath, command);
    this.interruptGracePeriodUnits = interruptGracePeriodUnits;
    this.interruptGracePeriodValue = interruptGracePeriodValue;
  }

  /**
   * This works fine for commands that return in a finite amount of time. Lastly, it waits until the
   * sshCommand is done before processing the input / output.
   *
   * @throws IOException
   */
  @Override
  protected void doExecute(SSHClient sshClient, Session session)
      throws IOException, InterruptedException {
    session.allocateDefaultPTY();
    try (Session.Command sshCommand = session.exec(command.getShellCommand().getCommandLine())) {
      final SSHCommandProcessExecution sshCommandProcessExecution =
          new SSHCommandProcessExecution(
              sshCommand.getInputStream(),
              sshCommand.getErrorStream(),
              sshCommand.getOutputStream(),
              command,
              sshCommand,
              interruptGracePeriodUnits,
              interruptGracePeriodValue);
      sshCommandProcessExecution.run();
    }
  }
}
