package wf.utils.db.table;



import lombok.AccessLevel;
import lombok.Getter;
import lombok.SneakyThrows;
import wf.utils.db.entity.DBEntity;
import wf.utils.db.models.ImprovedPreparedStatement;
import wf.utils.db.models.DBConnection;

import java.sql.ResultSet;
import java.sql.Statement;
import java.util.*;

@Getter
public abstract class AbstractDBTable<E extends DBEntity<ID>, ID> implements DBTable<E, ID> {


    private final DBConnection dbConnection;
    private final List<String> columns;

    @Getter(AccessLevel.NONE)
    private final String columnsSqlForSave;
    @Getter(AccessLevel.NONE)
    private final String columnsSqlForUpdate;
    @Getter(AccessLevel.NONE)
    private final String columnsSql;
    @Getter(AccessLevel.NONE)
    private final String columnsValuesSql;


    public AbstractDBTable(DBConnection dbConnection) {
        this.dbConnection = dbConnection;
        this.columns = getColumns();
        this.columnsSql = getColumnsSql(columns);
        this.columnsSqlForSave = getColumnsSqlForSave(columns);
        this.columnsSqlForUpdate = getColumnsSqlForUpdate(columns);
        this.columnsValuesSql = columnsValuesSql(columns);

        createTable();
    }


    @Override
    @SneakyThrows
    public Optional<E> findById(ID id) {
        ImprovedPreparedStatement ps = preparedStatement("SELECT %c FROM %t WHERE id=?");

        ps.setId(1, id);

        ResultSet rs = ps.executeQuery();

        if(!rs.next())
            return Optional.empty();

        System.out.println(rs.getRow());


        return Optional.of(parse(rs));
    }


    @Override
    @SneakyThrows
    public boolean deleteById(ID id) {
        ImprovedPreparedStatement ps = preparedStatement("DELETE FROM %t WHERE id=?");

        ps.setId(1, id);

        return ps.execute();
    }


    @Override
    @SneakyThrows
    public boolean update(E entity) {
        if(entity.getId() == null)
            throw new NullPointerException("ID cannot be null!");

        ImprovedPreparedStatement ps = preparedStatementForUpdate("UPDATE %t SET %cfu WHERE id=?");

        parse(entity, ps);
        ps.setId(ps.getParameterMetaData().getParameterCount(), entity.getId());

        return ps.executeUpdate() != 0;
    }



    @Override
    @SneakyThrows
    @SuppressWarnings("unchecked")
    public E save(E entity) {
        ImprovedPreparedStatement ps = preparedStatementForSave("INSERT INTO %t(%cfs) VALUES (%v)");

        parse(entity, ps);


        ps.executeUpdate();

        ResultSet generatedKeys = ps.getGeneratedKeys();

        if (generatedKeys.next()) {
            if(entity.getIdClass() == Integer.class)
                entity.setId((ID) (Integer) generatedKeys.getInt(1));
            else if(entity.getIdClass() == Long.class)
                entity.setId((ID) (Long) generatedKeys.getLong(1));
            else if(entity.getIdClass() == UUID.class)
                entity.setId((ID) UUID.fromString(generatedKeys.getString(1)));
            else if(entity.getIdClass() == String.class)
                entity.setId((ID) generatedKeys.getString(1));
        }

        return entity;
    }

    public ImprovedPreparedStatement preparedStatement(String sql) {
        return preparedStatement(sql, 0);
    }

    public ImprovedPreparedStatement preparedStatement(String sql, int autogeneratedKeys) {
        return new ImprovedPreparedStatement(dbConnection.preparedStatement(
                sql
                        .replace("%t", getTableName())
                        .replace("%c", columnsSql)
            , autogeneratedKeys));
    }


    public ImprovedPreparedStatement preparedStatementForSave(String sql) {
        return preparedStatement(sql
                .replace("%cfs", columnsSqlForSave)
                .replace("%v", columnsValuesSql)
            , Statement.RETURN_GENERATED_KEYS);
    }

    public ImprovedPreparedStatement preparedStatementForUpdate(String sql) {
        return preparedStatement(sql
                .replace("%cfu", columnsSqlForUpdate)
        );
    }

    public ImprovedPreparedStatement preparedStatementForCreateTable(String sql) {
        return preparedStatement(sql
                .replace("%id_integer", "id INT PRIMARY KEY GENERATED ALWAYS AS IDENTITY (START WITH 1 INCREMENT BY 1)")
                .replace("%id_long", "id BIGINT PRIMARY KEY GENERATED ALWAYS AS IDENTITY (START WITH 1 INCREMENT BY 1)")
        );
    }

    @SneakyThrows
    public void createTable() {
        ImprovedPreparedStatement ps = preparedStatementForCreateTable(getCreateTableSql());

        ps.execute();
    }

    private static String getColumnsSqlForSave(List<String> columns) {
        StringJoiner joiner = new StringJoiner(",");

        for(String column : columns)
            if(!column.equals("id"))
                joiner.add(column);

        return joiner.toString();
    }

    private static String getColumnsSqlForUpdate(List<String> columns) {
        StringJoiner joiner = new StringJoiner(",");

        for(String column : columns)
            if(!column.equals("id"))
                joiner.add(column + "=?");

        return joiner.toString();
    }

    private static String getColumnsSql(List<String> columns) {
        StringJoiner joiner = new StringJoiner(",");

        for(String column : columns)
            joiner.add(column);

        return joiner.toString();
    }

    private static String columnsValuesSql(List<String> columns) {
        StringJoiner joiner = new StringJoiner(",");

        for(String column : columns)
            if(!column.equals("id"))
                joiner.add("?");

        return joiner.toString();
    }


    public abstract List<String> getColumns();

    public abstract String getCreateTableSql();


}
