package com.xmfuncoding.fundialog

import android.content.res.Resources
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Bundle
import android.text.TextPaint
import android.text.method.LinkMovementMethod
import android.text.style.ClickableSpan
import android.text.style.ForegroundColorSpan
import android.util.DisplayMetrics
import android.view.*
import android.view.View.OnClickListener
import androidx.annotation.StringRes
import androidx.core.content.ContextCompat
import androidx.core.text.set
import androidx.core.text.toSpannable
import androidx.core.view.isVisible
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.FragmentTransaction
import com.xmfuncoding.fundialog.databinding.FunDialogBinding

/**
 *  Author:      XmFunCoding
 *  Email:       xmfuncoding@163.com
 *  Date:        2022-10-04 16:30
 *  Description: 统一对话框
 */
class FunDialog : DialogFragment() {

    companion object {
        const val TITLE_KEY         = "title"
        const val MESSAGE_KEY       = "message"
        const val BOLD_MESSAGE_KEY  = "boldMessage"
        const val CONFIRM_TEXT_KEY  = "confirmText"
        const val CANCEL_TEXT_KEY   = "cancelText"

        const val TITLE_RES_KEY         = "titleRes"
        const val MESSAGE_RES_KEY       = "messageRes"
        const val BOLD_MESSAGE_RES_KEY  = "boldMessageRes"
        const val CONFIRM_TEXT_RES_KEY  = "confirmTextRes"
        const val CANCEL_TEXT_RES_KEY   = "cancelTextRes"

        const val CANCELABLE_KEY    = "cancelable"

        const val NO_TEXT = 0

        fun newInstance(
            // title、titleRes 二选一
            title: String? = "",
            @StringRes titleRes: Int = NO_TEXT,

            // message、messageRes 二选一
            message: String? = "",
            @StringRes messageRes: Int = NO_TEXT,

            boldMessage: ArrayList<String> = arrayListOf(),
            boldMessageRes: ArrayList<Int> = arrayListOf(),

            // confirmText、confirmTextRes 二选一
            confirmText: String? = "",
            @StringRes confirmTextRes: Int = NO_TEXT,

            // cancelText、cancelTextRes 二选一
            cancelText: String? = "",
            @StringRes cancelTextRes: Int = NO_TEXT,

            cancelable: Boolean = true,
        ): FunDialog {
            val fragment = FunDialog()
            val bundle = Bundle()
            bundle.putString(TITLE_KEY, title)
            bundle.putInt(TITLE_RES_KEY, titleRes)

            bundle.putString(MESSAGE_KEY, message)
            bundle.putInt(MESSAGE_RES_KEY, messageRes)

            bundle.putStringArrayList(BOLD_MESSAGE_KEY, boldMessage)
            bundle.putIntegerArrayList(BOLD_MESSAGE_RES_KEY, boldMessageRes)

            bundle.putString(CONFIRM_TEXT_KEY, confirmText)
            bundle.putInt(CONFIRM_TEXT_RES_KEY, confirmTextRes)

            bundle.putString(CANCEL_TEXT_KEY, cancelText)
            bundle.putInt(CANCEL_TEXT_RES_KEY, cancelTextRes)

            bundle.putBoolean(CANCELABLE_KEY, cancelable)
            fragment.arguments = bundle
            return fragment
        }
    }

    private lateinit var binding: FunDialogBinding

    private var mTitle: String? = null
    private var mMessage: String? = null
    private var mBoldMessage: List<String>? = null
    private var mConfirmText: String? = null
    private var mCancelText: String? = null
    private var mCancelable = false

    private var mOnBoldMessageClickListenerList: List<OnClickListener>? = null

    private var mOnConfirmClickListener: OnClickListener? = null
    private var mOnCancelClickListener: OnClickListener? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?,
    ): View {
        binding = FunDialogBinding.inflate(layoutInflater)
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        mTitle = arguments?.getString(TITLE_KEY)
        mMessage = arguments?.getString(MESSAGE_KEY)
        mBoldMessage = arguments?.getStringArrayList(BOLD_MESSAGE_KEY)
        mConfirmText = arguments?.getString(CONFIRM_TEXT_KEY)
        mCancelText = arguments?.getString(CANCEL_TEXT_KEY)

        if (mTitle.isNullOrEmpty()) {
            mTitle = arguments?.getInt(TITLE_RES_KEY)?.let { context?.getString(it) }
        }
        if (mMessage.isNullOrEmpty()) {
            mMessage = arguments?.getInt(MESSAGE_RES_KEY)?.let { context?.getString(it) }
        }
        if (mBoldMessage.isNullOrEmpty()) {
            val intList = arguments?.getIntegerArrayList(BOLD_MESSAGE_RES_KEY)
            val stringList = mutableListOf<String>()
            intList?.forEach {
                context?.getString(it)?.let { it1 -> stringList.add(it1) }
            }
            mBoldMessage = stringList
        }
        if (mConfirmText.isNullOrEmpty()) {
            mConfirmText = arguments?.getInt(CONFIRM_TEXT_RES_KEY)?.let { context?.getString(it) }
        }
        if (mCancelText.isNullOrEmpty()) {
            mCancelText = arguments?.getInt(CANCEL_TEXT_RES_KEY)?.let { context?.getString(it) }
        }

        mCancelable = arguments?.getBoolean(CANCELABLE_KEY) ?: true

        isCancelable = mCancelable

        // 标题
        if (mTitle.isNullOrEmpty()) {
            binding.tvTitle.isVisible = false
        } else {
            binding.tvTitle.isVisible = true
            binding.tvTitle.text = mTitle
        }

        // 内容
        if (!mBoldMessage.isNullOrEmpty() && !mMessage.isNullOrEmpty()) {
            val spannable = mMessage!!.toSpannable()

            mBoldMessage?.forEachIndexed { index, item ->
                val start = mMessage!!.indexOf(item)
                val end = start + item.length

                if (start >= 0) {
                    spannable[start, end] = object : ClickableSpan() {
                        override fun onClick(view: View) {
                            mOnBoldMessageClickListenerList?.get(index)?.onClick(view)
                        }

                        // 去除下划线
                        override fun updateDrawState(ds: TextPaint) {
                            super.updateDrawState(ds)
                            ds.color = ds.linkColor
                            ds.isUnderlineText = false
                        }
                    }

                    // 字体颜色
                    spannable[start, end] = ForegroundColorSpan(ContextCompat.getColor(requireActivity(), R.color.colorDialogBoldText))
                }
            }

            binding.tvMessage.text = spannable
            binding.tvMessage.movementMethod = LinkMovementMethod.getInstance() // 设置了才能点击
        } else {
            binding.tvMessage.text = mMessage
        }

        binding.btnCancel.isVisible = !mCancelText.isNullOrEmpty()
        binding.viewDivider.isVisible = !mCancelText.isNullOrEmpty()

        binding.btnCancel.text = mCancelText
        binding.btnConfirm.text = mConfirmText

        binding.btnCancel.setOnClickListener {
            mOnCancelClickListener?.onClick(it)
            dismiss()
        }

        binding.btnConfirm.setOnClickListener {
            mOnConfirmClickListener?.onClick(it)
            dismiss()
        }
    }

    override fun onStart() {
        /*
        因为View在添加后,对话框最外层的ViewGroup并不知道我们导入的View所需要的的宽度。 所以我们需要在onStart生命周期里修改对话框尺寸参数
        */
        super.onStart()

        val width = resources.displayMetrics.widthPixels
        val height = resources.displayMetrics.heightPixels

        val window = requireDialog().window
        val attributes = window?.attributes

        if (width < height) {
            // 竖屏
            attributes?.width = (width * 0.9f).toInt()
            attributes?.height = (height * 0.8f).toInt()
        } else {
            // 横屏
            attributes?.width = dp2px(360F).toInt()
            attributes?.height = (height * 0.9f).toInt()
        }

        attributes?.gravity = Gravity.CENTER
        window?.attributes = attributes
        window?.decorView?.background = ColorDrawable(Color.TRANSPARENT)
    }

    fun setOnBoldMessageClickListenerList(onBoldMessageClickListenerList: List<OnClickListener>?) {
        mOnBoldMessageClickListenerList = onBoldMessageClickListenerList
    }

    fun setOnConfirmClickListener(onConfirmClickListener: View.OnClickListener?) {
        mOnConfirmClickListener = onConfirmClickListener
    }

    fun setOnCancelClickListener(onCancelClickListener: View.OnClickListener?) {
        mOnCancelClickListener = onCancelClickListener
    }

    fun show(manager: FragmentManager) {
        show(manager, "dialog")
    }

    override fun show(manager: FragmentManager, tag: String?) {
        try {
            val ft: FragmentTransaction = manager.beginTransaction()
            ft.add(this, tag)
            ft.commitAllowingStateLoss()
        } catch (e: IllegalStateException) {
            e.printStackTrace()
        }
    }

    fun isShowing() : Boolean {
        return dialog?.isShowing ?: false
    }

    /**
     * Value of dp to value of px.
     *
     * @param dpValue The value of dp.
     * @return value of px
     */
    fun dp2px(dpValue: Float): Float {
        val scale = resources.displayMetrics.density
        return dpValue * scale + 0.5f
    }
}