package com.livehelps.dify.api;

import com.livehelps.dify.service.DifyCompletionApiService;

import lombok.extern.slf4j.Slf4j;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Response;

import static com.livehelps.dify.DIfyApiServiceGenerator.createService;

import java.io.IOException;

import com.livehelps.dify.api.callback.CompletionStreamCallback;
import com.livehelps.dify.api.callback.StreamEventDispatcher;
import com.livehelps.dify.data.enums.AnnotationReplyAction;
import com.livehelps.dify.data.request.AnnotationReplyRequest;
import com.livehelps.dify.data.request.AnnotationRequest;
import com.livehelps.dify.data.request.CompletionMessageRequest;
import com.livehelps.dify.data.request.FeedBackRequest;
import com.livehelps.dify.data.request.StopCompletionMessageRequest;
import com.livehelps.dify.data.request.TextToAudioRequest;
import com.livehelps.dify.data.response.Annotation;
import com.livehelps.dify.data.response.AnnotationListResponse;
import com.livehelps.dify.data.response.AnnotationReplyResponse;
import com.livehelps.dify.data.response.ChatCompletionResponse;
import com.livehelps.dify.data.response.FeedBackListResponse;
import com.livehelps.dify.data.response.LittleResponse;
import com.livehelps.dify.data.response.StopCompletionMessageResponse;

import static com.livehelps.dify.DIfyApiServiceGenerator.*;

@Slf4j
public class DIfyCompletionApi {

    private final DifyCompletionApiService completionApiService;


    public DIfyCompletionApi(String baseUrl, String apiKey) {
        this.completionApiService = createService(DifyCompletionApiService.class,apiKey,baseUrl);;
    }

    public ChatCompletionResponse sendCompletionMessage(CompletionMessageRequest message){
        log.debug("send completion message: {}", message);
        return executeSync(completionApiService.sendCompletionMessage(message));
    }

    public void sendCompletionMessageStream(CompletionMessageRequest message, CompletionStreamCallback callback){
        log.debug("send completion message stream: {}", message);
        Call<ResponseBody> call = completionApiService.sendCompletionMessageStream(message);
        executeStreamRequest(call,(line) -> processStreamLine(line, callback, (data, eventType) -> {
            StreamEventDispatcher.dispatchCompletionEvent(callback, data);
        }), callback::onException);
    }


    public StopCompletionMessageResponse stopCompletionMessage(String taskId, StopCompletionMessageRequest request){
        log.debug("stop completion message taskId {} request: {}", taskId,request);
        return executeSync(completionApiService.stopCompletionMessage(taskId,request));
    }


    public LittleResponse feedbackMessage(String messageId, FeedBackRequest request){
        log.debug("feedback completion message messageId {} request: {}", messageId,request);
        return executeSync(completionApiService.feedbackMessage(messageId,request));
    }

    public FeedBackListResponse getFeedBackList(Integer page,Integer limit){
        log.debug("get feed back list page {} limit: {}", page,limit);
        return executeSync(completionApiService.getAppfeedbacks(page,limit));
    }

    public byte[] textToAudio(TextToAudioRequest request) throws IOException{
        ResponseBody responseBody = executeSync(completionApiService.textToAudio(request));
        return responseBody.bytes();
    }

    public Annotation createAnnotation(AnnotationRequest request){
        log.debug("create annotation request: {}", request);
        return executeSync(completionApiService.createAnnotation(request));
    }

    public AnnotationListResponse getAnnotationList(Integer page,Integer limit){
        log.debug("get annotation list page {} limit: {}", page,limit);
        return executeSync(completionApiService.getAnnotationList(page,limit));
    }

    public void deleteAnnotation(String id){
        log.debug("delete annotation id: {}", id);
        executeSync(completionApiService.deleteAnnotation(id));
    }

    public Annotation updateAnnotation(String id,AnnotationRequest request){
        log.debug("update annotation id: {} request: {}", id,request);
        return executeSync(completionApiService.updateAnnotation(id,request));
    }

    public AnnotationReplyResponse annotationReply(AnnotationReplyAction action,AnnotationReplyRequest request){
        log.debug("annotation reply action: {} request: {}", action,request);
        return executeSync(completionApiService.annotationReply(action,request));
    }

    public AnnotationReplyResponse getAnnotationReplyStatus(AnnotationReplyAction action,String jobId){
        log.debug("get annotation reply status action: {} jobId: {}", action,jobId);
        return executeSync(completionApiService.getAnnotationReplyStatus(action,jobId));
    }
}
