/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.exec;

import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.inject.Binder;
import com.google.inject.Key;
import com.google.inject.name.Names;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigObject;
import com.typesafe.config.ConfigValue;
import com.typesafe.config.ConfigValueFactory;
import com.typesafe.config.ConfigValueType;
import org.jooby.Env;
import org.jooby.Jooby.Module;
import org.jooby.funzy.Throwing;
import org.jooby.funzy.Try;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinPool.ForkJoinWorkerThreadFactory;
import java.util.concurrent.ForkJoinWorkerThread;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

/**
 * <h1>executor</h1>
 * <p>
 * Manage the life cycle of {@link ExecutorService} and build async apps, schedule tasks, etc...
 * </p>
 *
 * <h2>usage</h2>
 *
 * <pre>{@code
 * ...
 * import org.jooby.exec.Exec;
 * ...
 *
 * {
 *   use(new Exec());
 *
 *   get("/", req -> {
 *     ExecutorService executor = req.require(ExecutorService.class);
 *     // work with executor
 *   });
 * }
 * }</pre>
 *
 * <p>
 * The default executor is a {@link Executors#newFixedThreadPool(int)} with threads defined by
 * {@link Runtime#availableProcessors()}
 * </p>
 *
 * <h2>explicit creation</h2>
 * <p>
 * The default {@link ExecutorService} is nice and give you something that just works out of the
 * box. But, what if you need to control the number of threads?
 * </p>
 * <p>
 * Explicit control is provided via <code>executors</code> which allow the following syntax:
 * </p>
 *
 * <pre>
 * type (= int)? (, daemon (= boolean)? )? (, priority (= int)? )?
 * </pre>
 *
 * <p>
 * Let's see some examples:
 * </p>
 *
 * <pre>
 * # fixed thread pool with a max number of threads equals to the available runtime processors
 * executors = "fixed"
 * </pre>
 *
 * <pre>
 * # fixed thread pool with a max number of 10 threads
 * executors = "fixed = 10"
 * </pre>
 *
 * <pre>
 * # fixed thread pool with a max number of 10 threads
 * executors = "fixed = 10"
 * </pre>
 *
 * <pre>
 * # scheduled thread pool with a max number of 10 threads
 * executors = "scheduled = 10"
 * </pre>
 *
 * <pre>
 * # cached thread pool with daemon threads and max priority
 * executors = "cached, daemon = true, priority = 10"
 * </pre>
 *
 * <pre>
 * # forkjoin thread pool with asyncMode
 * executors = "forkjoin, asyncMode = true"
 * </pre>
 *
 * <h2>multiple executors</h2>
 * <p>
 * Multiple executors are provided by expanding the <code>executors</code> properties, like:
 * </p>
 *
 * <pre>
 *  executors {
 *    pool1: fixed
 *    jobs: forkjoin
 *  }
 * </pre>
 *
 * <p>
 * Later, you can request your executor like:
 * </p>
 * <pre>{@code
 * {
 *   get("/", req -> {
 *     ExecutorService pool1 = req.require("pool1", ExecutorService.class);
 *     ExecutorService jobs = req.require("jobs", ExecutorService.class);
 *   });
 * }
 * }</pre>
 *
 * <h2>shutdown</h2>
 * <p>
 * Any {@link ExecutorService} created by this module will automatically shutdown on application
 * shutdown time.
 * </p>
 *
 * @author edgar
 * @since 0.16.0
 */
public class Exec implements Module {

  private static final BiConsumer<String, Executor> NOOP = (n, e) -> {
  };

  /** The logging system. */
  private final Logger log = LoggerFactory.getLogger(getClass());

  private boolean daemon = true;

  private int priority = Thread.NORM_PRIORITY;

  private final Map<String, Throwing.Function4<String, Integer, Supplier<ThreadFactory>, Map<String, Object>, ExecutorService>> f =
      /** executor factory. */
      ImmutableMap
          .of(
              "cached", (name, n, tf, opts) -> Executors.newCachedThreadPool(tf.get()),
              "fixed", (name, n, tf, opts) -> Executors.newFixedThreadPool(n, tf.get()),
              "scheduled", (name, n, tf, opts) -> Executors.newScheduledThreadPool(n, tf.get()),
              "forkjoin", (name, n, tf, opts) -> {
                boolean asyncMode = Boolean.parseBoolean(opts.getOrDefault("asyncMode", "false")
                    .toString());
                return new ForkJoinPool(n, fjwtf(name), null, asyncMode);
              });

  private final String namespace;

  protected Exec(final String namespace) {
    this.namespace = namespace;
  }

  public Exec() {
    this("executors");
  }

  /**
   * Defined the default value for daemon. This value is used when a executor spec doesn't define a
   * value for daemon. Default is: <code>true</code>
   *
   * @param daemon True for default daemon.
   * @return This module.
   */
  public Exec daemon(final boolean daemon) {
    this.daemon = daemon;
    return this;
  }

  /**
   * Defined the default value for priority. This value is used when a executor spec doesn't define
   * a value for priority. Default is: {@link Thread#NORM_PRIORITY}.
   *
   * @param priority One of {@link Thread#MIN_PRIORITY}, {@link Thread#NORM_PRIORITY} or
   *        {@link Thread#MAX_PRIORITY}.
   * @return This module.
   */
  public Exec priority(final int priority) {
    this.priority = priority;
    return this;
  }

  @Override
  public Config config() {
    return ConfigFactory.empty("exec.conf").withValue(namespace,
        ConfigValueFactory.fromAnyRef("fixed"));
  }

  @Override
  public void configure(final Env env, final Config conf, final Binder binder) {
    configure(env, conf, binder, NOOP);
  }

  protected void configure(final Env env, final Config conf, final Binder binder,
      final BiConsumer<String, Executor> callback) {
    List<Map<String, Object>> executors = conf.hasPath(namespace)
        ? executors(conf.getValue(namespace), daemon, priority,
        Runtime.getRuntime().availableProcessors())
        : Collections.emptyList();
    List<Entry<String, ExecutorService>> services = new ArrayList<>(executors.size());
    for (Map<String, Object> options : executors) {
      // thread factory options
      String name = (String) options.remove("name");
      log.debug("found executor: {}{}", name, options);
      Boolean daemon = (Boolean) options.remove("daemon");
      Integer priority = (Integer) options.remove("priority");
      String type = String.valueOf(options.remove("type"));
      // number of processors
      Integer n = (Integer) options.remove(type);

      // create executor
      Throwing.Function4<String, Integer, Supplier<ThreadFactory>, Map<String, Object>, ExecutorService> factory = f
          .get(type);
      if (factory == null) {
        throw new IllegalArgumentException(
            "Unknown executor: " + type + " must be one of " + f.keySet());
      }
      ExecutorService executor = factory.apply(type, n, () -> factory(name, daemon, priority),
          options);

      bind(binder, name, executor);
      callback.accept(name, executor);

      services.add(Maps.immutableEntry(name, executor));
    }

    services.stream()
        .filter(it -> it.getKey().equals("default"))
        .findFirst()
        .ifPresent(e -> bind(binder, null, e.getValue()));

    env.onStop(() -> {
      services.forEach(exec -> Try.run(() -> exec.getValue().shutdown()).onFailure(cause -> log.error("shutdown of {} resulted in error", exec.getKey(), cause)));
      services.clear();
    });
  }

  @SuppressWarnings({"rawtypes", "unchecked"})
  private static void bind(final Binder binder, final String name, final ExecutorService executor) {
    Class klass = executor.getClass();

    Set<Class> types = collector(klass);
    for (Class type : types) {
      Key key = name == null ? Key.get(type) : Key.get(type, Names.named(name));
      binder.bind(key).toInstance(executor);
    }
  }

  @SuppressWarnings("rawtypes")
  private static Set<Class> collector(final Class type) {
    if (type != null && Executor.class.isAssignableFrom(type)) {
      Set<Class> types = new HashSet<>();
      if (type.isInterface() || !Modifier.isAbstract(type.getModifiers())) {
        types.add(type);
      }
      types.addAll(collector(type.getSuperclass()));
      Arrays.asList(type.getInterfaces()).forEach(it -> types.addAll(collector(it)));
      return types;
    }
    return Collections.emptySet();
  }

  private static ThreadFactory factory(final String name, final boolean daemon,
      final int priority) {
    AtomicLong id = new AtomicLong(0);
    return r -> {
      Thread thread = new Thread(r, name + "-" + id.incrementAndGet());
      thread.setDaemon(daemon);
      thread.setPriority(priority);
      return thread;
    };
  }

  private static ForkJoinWorkerThreadFactory fjwtf(final String name) {
    AtomicLong id = new AtomicLong();
    return pool -> {
      ForkJoinWorkerThread thread = ForkJoinPool.defaultForkJoinWorkerThreadFactory.newThread(pool);
      thread.setName(name + "-" + id.incrementAndGet());
      return thread;
    };
  }

  private static List<Map<String, Object>> executors(final ConfigValue candidate,
      final boolean daemon, final int priority, final int n) {
    if (candidate.valueType() == ConfigValueType.STRING) {
      Map<String, Object> options = executor("default", daemon, priority, n,
          candidate.unwrapped());
      return ImmutableList.of(options);
    }
    ConfigObject conf = (ConfigObject) candidate;
    List<Map<String, Object>> result = new ArrayList<>();
    for (Entry<String, ConfigValue> executor : conf.entrySet()) {
      String name = executor.getKey();
      Object value = executor.getValue().unwrapped();
      Map<String, Object> options = new HashMap<>(executor(name, daemon, priority, n, value));
      result.add(options);
    }
    return result;
  }

  @SuppressWarnings("unchecked")
  private static Map<String, Object> executor(final String name, final boolean daemon,
      final int priority,
      final int n,
      final Object value) {
    Map<String, Object> options = new HashMap<>();
    options.put("name", name);
    options.put("daemon", daemon);
    options.put("priority", priority);
    if (value instanceof Map) {
      Map<String, Object> config = (Map<String, Object>) value;
      Object rawType = config.get("type");
      if (rawType == null) {
        throw new IllegalArgumentException("Missing executor type");
      }
      String type = rawType.toString();
      options.put("type", type);
      options.put(type, config.containsKey("size") ?
          Integer.parseInt(config.get("size").toString()) : n);
      options.put("daemon", config.containsKey("daemon") ?
          Boolean.parseBoolean(config.get("daemon").toString()) : daemon);
      options.put("asyncMode", config.containsKey("asyncMode") && Boolean.parseBoolean(config.get("asyncMode").toString()));
      options.put("priority", config.containsKey("priority") ?
          Integer.parseInt(config.get("priority").toString()) : priority);
    } else {
      Iterable<String> spec = Splitter.on(",").trimResults().omitEmptyStrings()
          .split(value.toString());
      for (String option : spec) {
        String[] opt = option.split("=");
        String optname = opt[0].trim();
        Object optvalue;
        switch (optname) {
          case "daemon":
            optvalue = opt.length > 1 ? Boolean.parseBoolean(opt[1].trim()) : daemon;
            break;
          case "asyncMode":
            optvalue = opt.length > 1 && Boolean.parseBoolean(opt[1].trim());
            break;
          case "priority":
            optvalue = opt.length > 1 ? Integer.parseInt(opt[1].trim()) : priority;
            break;
          default:
            optvalue = opt.length > 1 ? Integer.parseInt(opt[1].trim()) : n;
            options.put("type", optname);
            break;
        }
        options.put(optname, optvalue);
      }
    }
    return options;
  }

}
